﻿'VB.vb : STSeries example in VB.NET

Imports Scriptel.STSeries

Module VB
    Public Class STExample
        Private device As STSeriesDevice
        Private sig As STSignature
        Private wait As Threading.EventWaitHandle
        Private containerTarget As STSeriesDevice.ContainerSelectedEventHandler
        Private buttonTarget As STSeriesDevice.ButtonSelectedEventHandler
        Private swipeTarget As STCardReader.CardSwipedEventHandler
        Private unpluggedTarget As EventHandler
        Private unpluggedTimer As System.Threading.Timer
        Private devices As ISTSeriesDevicesManager = New STSeriesDevicesManager
        Private reading As Boolean
        Private reconnect As Boolean

        ''' <summary>
        ''' Initializes a new instance of the <see cref="STExample"/> class.
        ''' </summary>
        Public Sub New()
            wait = New Threading.EventWaitHandle(False, Threading.EventResetMode.ManualReset)
            reading = True
            reconnect = False
        End Sub

        ''' <summary>
        ''' Gets the first Scriptel STSeries device found.
        ''' Reports all devices found to stdout
        ''' </summary>
        ''' <returns></returns>
        Public Function getFirstDevice() As ISTSeriesDevice
            Dim firstDevice As ISTSeriesDevice = Nothing
            Dim numOfDevices As Integer = devices.GetNumberOfSTSeriesDevices()

            If numOfDevices = 0 Then
                Console.WriteLine("No Scriptel ScripTouch Series devices were found on the system.")
            ElseIf numOfDevices = 1 Then
                Console.WriteLine("The following Scriptel ScripTouch Series device has been found:")
            Else
                Console.WriteLine("The following Scriptel ScripTouch Series devices have been found:")
            End If

            For devNum As Integer = 0 To numOfDevices - 1
                Dim device As ISTSeriesDevice = devices.GetSTSeriesDevice(devNum)
                If devNum = 0 Then
                    firstDevice = device
                End If

                Console.WriteLine("{0}: {1}", devNum + 1, device.DeviceName)
            Next

            Console.WriteLine()

            Return firstDevice

        End Function

        ''' <summary>
        ''' Called when the screen is touched (not starting on a button container).
        ''' </summary>
        ''' <param name="sender">The sender.</param>
        ''' <param name="e">The <see cref="STSeriesDevice.RegionSelectedEventArgs"/> instance containing the event data.</param>
        Sub OnContainerEvent(sender As Object, e As STSeriesDevice.RegionSelectedEventArgs)
            'add the point to the signature
            sig.AddPoint(e.SigPoint)

            'write the coordinates to stdout
            Dim coordFomrat As String = "{0,5}"
            Dim stringOut As New System.Text.StringBuilder("Coordinates ")
            stringOut.AppendFormat(coordFomrat, e.SigPoint.X)
            stringOut.Append(", ")
            stringOut.AppendFormat(coordFomrat, e.SigPoint.Y)
            If e.SigPoint.SwitchStatus = STSignaturePoint.InputDeviceStatus.PenTipSwitchFingerDown Then
                stringOut.Append(" pendown")
            Else
                stringOut.Append(" penup  ")
            End If

            Console.WriteLine(stringOut)

        End Sub

        ''' <summary>
        ''' Called when the touch starts on a button.
        ''' </summary>
        ''' <param name="sender">The sender.</param>
        ''' <param name="e">The <see cref="STSeriesDevice.RegionSelectedEventArgs"/> instance containing the event data.</param>
        Sub OnButtonEvent(sender As Object, e As STSeriesDevice.RegionSelectedEventArgs)
            If e.RegionNumber = 3 Then  'OK button
                sig = New STSignature   'forget the old signature
                device.CurrentDisplay.ClearDisplay()
            ElseIf e.RegionNumber = 4 Then
                reading = False 'Let the main thread know it's time to be done.
                wait.Set() 'Cancel pressed, allow RunFirstDevice to 
            End If
        End Sub

        ''' <summary>
        ''' Called when a magnetic strip card is swiped.
        ''' </summary>
        ''' <param name="sender">The sender.</param>
        ''' <param name="e">The <see cref="STCardReader.CardSwipedEventArgs"/> instance containing the event data.</param>
        Sub OnCardSwipeEvent(sender As Object, e As STCardReader.CardSwipedEventArgs)
            Console.WriteLine(e.TrackOneContent)
        End Sub

        ''' <summary>
        ''' Called when the currently open device is unplugged or otherwise removed from the system.
        ''' </summary>
        ''' <param name="sender">The sender.</param>
        ''' <param name="e">The event.</param>
        ''' <remarks></remarks>
        Sub OnUnplugEvent(sender As Object, e As EventArgs)
            'Tell the main loop to reconnect to the device when possible.
            reconnect = True

            'Find our unplugged tick callback.
            Dim unpluggedCallback As New System.Threading.TimerCallback(AddressOf UnplugTimerTick)

            'Create a new timer, have it call the unpluggedCallback every 500ms after waiting 1000ms initially.
            unpluggedTimer = New System.Threading.Timer(unpluggedCallback, Nothing, 1000, 500)

            Console.WriteLine("Device has been unplugged, waiting for a new device.")
        End Sub

        ''' <summary>
        ''' This subroutine gets called regularly after a device has been disconnected
        ''' from the system. It checks for any available signature pads and will signal
        ''' the main loop when a new pad is found.
        ''' </summary>
        ''' <param name="state">State of the timer.</param>
        ''' <remarks></remarks>
        Sub UnplugTimerTick(ByVal state As Object)
            'Check to see if we have any devices.
            If devices.GetNumberOfSTSeriesDevices() > 0 Then
                'We have at least one device!

                'Dispose of our timer, we don't need it anymore.
                unpluggedTimer.Dispose()
                Console.WriteLine("Found a new device!")
                'Let the main thread know it's time to do something.
                wait.Set()
            Else
                'Nothing to do, no devices available.
                Console.WriteLine("Still waiting...")
            End If
        End Sub

        ''' <summary>
        ''' This subroutine opens the currently selected device.
        ''' This should only be called by the main thread.
        ''' </summary>
        ''' <remarks></remarks>
        Sub OpenDevice()
            device.Open()

            sig = New STSignature()
            device.StartReading()

            containerTarget = New STSeriesDevice.ContainerSelectedEventHandler(AddressOf OnContainerEvent)
            device.RegisterForContainerSelectedEvent(Me, containerTarget)
            buttonTarget = New STSeriesDevice.ButtonSelectedEventHandler(AddressOf OnButtonEvent)
            device.RegisterForButtonSelectedEvent(Me, buttonTarget)
            swipeTarget = New STCardReader.CardSwipedEventHandler(AddressOf OnCardSwipeEvent)
            device.CardReader.RegisterForCardSwipedEvent(Me, swipeTarget)
            unpluggedTarget = New EventHandler(AddressOf OnUnplugEvent)
            device.RegisterForDeviceUnpluggedEvent(Me, unpluggedTarget)
        End Sub

        ''' <summary>
        ''' This subroutine closes the currently selected device.
        ''' This should only be called by the main thread.
        ''' </summary>
        ''' <remarks></remarks>
        Sub CloseDevice()
            If device.IsOpened Then
                device.StopReading()
            End If
            device.Close()

            device.UnregisterForContainerSelectedEvent(Me, containerTarget)
            device.UnregisterForButtonSelectedEvent(Me, buttonTarget)
            device.CardReader.UnregisterForCardSwipedEvent(Me, swipeTarget)
            device.UnregisterForDeviceUnpluggedEvent(Me, unpluggedTarget)

            device = Nothing
            containerTarget = Nothing
            buttonTarget = Nothing
            swipeTarget = Nothing
            unpluggedTarget = Nothing
        End Sub

        ''' <summary>
        ''' Runs the first device.
        ''' </summary>
        Sub RunFirstDevice()
            device = getFirstDevice()

            If Not device Is Nothing Then
                Console.WriteLine("Using device #1")
                Console.WriteLine("Tap OK to clear screen")
                Console.WriteLine("Tap Cancel on screen to exit program" & vbNewLine)

                Try
                    OpenDevice()

                    While reading
                        'Block here until the event is signaled.
                        wait.WaitOne()
                        'Reset the event so we can block on it again next time around the loop.
                        wait.Reset()
                        'Check to see if we need to re-open the device.
                        If reading And reconnect Then
                            CloseDevice()

                            'We have at least one device! Let's open the first one.
                            device = devices.GetSTSeriesDevice(0)

                            OpenDevice()

                            reconnect = False
                        End If
                    End While

                    Console.WriteLine(vbNewLine & vbNewLine & "Cancel was pressed, exiting program")
                Catch ex As Exception
                    Console.WriteLine(vbNewLine & vbNewLine & "An Unexpected Exception happened")
                    Console.WriteLine(ex.Message)
                Finally
                    CloseDevice()
                End Try
            End If
        End Sub

    End Class

    Sub Main()
        Console.WriteLine("ScripTouch Series Example in VB.NET")
        'instantiate the class and run
        Dim example As STExample = New STExample
        example.RunFirstDevice()
    End Sub

End Module
