﻿// <copyright file="STSeriesExampleForm.cs" company="Scriptel Corporation">
//      Copyright 2013 - Scriptel Corporation
// </copyright>
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Drawing.Imaging;
using Scriptel.STSeries;

namespace Scriptel.STSeries.Examples.Csharp
{
    /// <summary>
    /// This class is a simple example of how to use the ST Series API
    /// to capture pen events.
    /// </summary>
    public partial class STSeriesExampleForm : Form
    {
        private Scriptel.STSeries.Examples.Csharp.STSeriesExample.CoordinateReceivedEventHandler coordinateTarget;
        private Scriptel.STSeries.Examples.Csharp.STSeriesExample.SwipeReceivedEventHandler swipeTarget;
        private Bitmap drawArea;
        private Scriptel.STSeries.Examples.Csharp.STSeriesExample example;
        private Pen inkPen = new Pen(Color.Black, (float)1.5);
        private Scriptel.STSeries.STSignaturePoint lastPoint = null;

        public STSeriesExampleForm()
        {
            InitializeComponent();

            example = new STSeriesExample();

            // Set some of the forms properties to increase drawing quality
            this.SetStyle(ControlStyles.AllPaintingInWmPaint | ControlStyles.UserPaint | ControlStyles.OptimizedDoubleBuffer, true);

            // create a bitmap of size 240 by 64
            drawArea = new Bitmap(240, 64, System.Drawing.Imaging.PixelFormat.Format32bppArgb);

            // Create a graphics object from the bitmap
            Graphics g = Graphics.FromImage(drawArea);

            // clear the graphics object, giving the drawn bitmap a solid background color.
            g.Clear(Color.White);

            // dispose of the graphics object to free resources
            g.Dispose();
        }

        private void OnSwipeReceived(object sender, STSeriesExample.SwipeReceivedEventArg e)
        {
            SetTextBox(e.CardSwipe.TrackOneContent, cardSwipeTextBox);
        }

        public void SetTextBox(string value, TextBox tb)
        {
            if (InvokeRequired)
            {
                this.Invoke(new Action<string, TextBox>(SetTextBox), new object[] { value, tb });
                return;
            }
            tb.Text = value;
        }


        private void OnCoordinateReceived(object sender, STSeriesExample.CoordinateReceivedEventArg e)
        {
            // if the point received isn't the first point in the line
            if (lastPoint != null)
            {
                // if the coordinate indicates the pen was picked up off the sensor
                if (e.SigPoint.SwitchStatus == STSignaturePoint.InputDeviceStatus.PenTipSwitchFingerUp)
                {
                    // designate the last point to null indicating the end of a line segment
                    lastPoint = null;
                }
                else
                {
                    // lock the draw area
                    lock (drawArea)
                    {
                        // create a graphics object from the bitmap
                        Graphics g = Graphics.FromImage(drawArea);

                        // Set some properties of the Graphics object
                        g.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.AntiAlias;
                        g.InterpolationMode = System.Drawing.Drawing2D.InterpolationMode.HighQualityBicubic;

                        // draw a line between the current point received and the last point received
                        // this line needs to be scaled from the coordinate system scale to the size of the bitmap
                        // using this formula

                        // (X - Xmin)
                        // ------------- * (BitmapWidth - 1)
                        // (Xmax - Xmin)

                        // since Xmin = 0 and Xmax = 65535 the equation can be simplified as shown below.
                        // this will scale a coordinate from 0 - 65535 to 0 - (Bitmap Width - 1)
                        g.DrawLine(
                                   inkPen,
                                   (int)Math.Round((e.SigPoint.X / (double)65535) * (drawArea.Width - 1)),
                                   (int)Math.Round((e.SigPoint.Y / (double)65535) * (drawArea.Height - 1)),
                                   (int)Math.Round((lastPoint.X / (double)65535) * (drawArea.Width - 1)),
                                   (int)Math.Round((lastPoint.Y / (double)65535) * (drawArea.Height - 1)));

                        // dispose of the graphics object
                        g.Dispose();
                    }
                    // force the form to redraw
                    this.Invalidate();

                    // designate the current point to be the last point received
                    lastPoint = e.SigPoint;
                }
            }
            else
            {
                // designate the current point to be the last point received
                lastPoint = e.SigPoint;
            }
        }

        private void OnPaint(object sender, PaintEventArgs e)
        {
            // Create a graphics object
            Graphics g = e.Graphics;

            // Set some properties of the Graphics object
            g.SmoothingMode = System.Drawing.Drawing2D.SmoothingMode.AntiAlias;
            g.InterpolationMode = System.Drawing.Drawing2D.InterpolationMode.HighQualityBicubic;

            // lock the bitmap
            lock (drawArea)
            {
                // draw the bitmap on the form
                g.DrawImage(drawArea, 15, 75, drawArea.Width, drawArea.Height);
            }
        }

        private void UI_button_SaveImage_Click(object sender, EventArgs e)
        {
            // Calls the method which saves the image of the signature to the desktop
            example.SaveSignatureToDesktop();
        }

        private void UI_Button_StartStop_Click(object sender, EventArgs e)
        {
            // See if the device has already been started.  If not
            if (!example.DeviceIsOpened)
            {
                // Check and see if the call to start the device succeeded
                try
                {
                    example.Start();
                    // Connect the event handler to the target method used to process the event
                    coordinateTarget = new STSeriesExample.CoordinateReceivedEventHandler(OnCoordinateReceived);
                    // register for an event to receive coordinate data
                    example.RegisterForCoordinateReceivedEvent(this, coordinateTarget);

                    swipeTarget = new STSeriesExample.SwipeReceivedEventHandler(OnSwipeReceived);

                    example.RegisterForSwipeReceivedEvent(this, swipeTarget);

                    // change the text on the button and the text on the label
                    uiButtonStartStop.Text = "Stop Device";
                    uiLabelConnectionStatus.Text = "Device Running";
                }
                catch (ConnectionException ce)
                {
                    uiLabelConnectionStatus.Text = ce.Message;
                }
            }
            else
            {
                // Unregister from the event used to receive coordinate data
                example.UnregisterForCoordinateReceivedEvent(this, coordinateTarget);

                // Stop the device
                example.Stop();

                // change the text on the button and the text on the label
                uiButtonStartStop.Text = "Start Device";
                uiLabelConnectionStatus.Text = "Device not Running";
            }
        }

        private void STSeriesExampleForm_Load(object sender, EventArgs e)
        {
        }
    }
}
