﻿// <copyright file="STSeriesExample.cs" company="Scriptel Corporation">
//      Copyright 2013 - Scriptel Corporation
// </copyright>
namespace Scriptel.STSeries.Examples.Csharp
{
    using System;
    using System.Drawing;
    using System.Drawing.Imaging;
    using Scriptel.STSeries;

    public class STSeriesExample
    {
        private ISTSeriesDevicesManager devManager;
        private STSeriesDevice.ButtonSelectedEventHandler buttonTarget;
        private STSeriesDevice.ContainerSelectedEventHandler containerTarget;
        private EventHandler unpluggedTarget;
        private STCardReader.CardSwipedEventHandler swipeTarget;
        private ISTSeriesDevice selectedDevice;
        private STSignature sig = new STSignature();
                
        public delegate void CoordinateReceivedEventHandler(object sender, CoordinateReceivedEventArg e);
        private event CoordinateReceivedEventHandler CoordinateEvent;
        public delegate void SwipeReceivedEventHandler(object sender, SwipeReceivedEventArg e);
        private event SwipeReceivedEventHandler SwipeEvent;

        /// <summary>
        /// Gets a value indicating whether a digitizer opened.
        /// </summary>
        /// <value>
        /// <c>true</c> if this instance contains a digitizer that is opened; otherwise, <c>false</c>.
        /// </value>
        public bool DeviceIsOpened
        {
            get
            {
                if ((this.selectedDevice != null) && this.selectedDevice.IsOpened)
                {
                    return true;
                }
                else
                {
                    return false;
                }
            }
        }

        /// <summary>
        /// Attaches to the first Scriptel digitizer found and start it
        /// </summary>
        public void Start()
        {
            // get all of the digitizers plugged into the system
            // here we will use the generic ISTSeriesDevice, rathan than figuring out which device it is
            // and using it specifically. An example of is show in Stop()
            this.devManager = new STSeriesDevicesManager();
            if (this.devManager.GetNumberOfSTSeriesDevices() > 0)
            {
                // Use the first device in the list
                this.selectedDevice = this.devManager.GetSTSeriesDevice(0);

                // Open the device to prepare it for use
                try
                {
                    this.selectedDevice.Open();
                }
                catch (STDeviceNotOpenedException dno)
                {
                    throw new ConnectionException(dno.Message);
                }
                
                // Register to receive information when a coordinate is received inside a button region
                this.buttonTarget = new STSeriesDevice.ButtonSelectedEventHandler(this.OnButtonSelected);
                this.selectedDevice.RegisterForButtonSelectedEvent(this, this.buttonTarget);

                // Register to receive information when a coordinate is received inside a container region
                this.containerTarget = new STSeriesDevice.ContainerSelectedEventHandler(this.OnContainerSelected);
                this.selectedDevice.RegisterForContainerSelectedEvent(this, this.containerTarget);

                // Register to receive notification when the device is unplugged
                this.unpluggedTarget = new EventHandler(this.OnDeviceUnplugged);
                this.selectedDevice.RegisterForDeviceUnpluggedEvent(this, this.unpluggedTarget);

                // Tell the device to start reading while putting the device in Pen mode
                this.selectedDevice.StartReading();

                this.swipeTarget = new STCardReader.CardSwipedEventHandler(this.OnCardSwipe);
                this.selectedDevice.CardReader.RegisterForCardSwipedEvent(this, this.swipeTarget);
            }
            else
            {
                throw new ConnectionException("No digitizers found");
            }
        }

        public void Stop()
        {
            // Check to see that you don't try and close a device that isn't opened
            if (this.DeviceIsOpened)
            {
                // Unregisters for the ButtonSelectedEvent.  This will stop you from receiving information when a button region is selected.
                this.selectedDevice.UnregisterForButtonSelectedEvent(this, this.buttonTarget);

                // Unregisters for the ContainerSelectedEvent.  This will stop you from receiving information when a container region is selected.
                this.selectedDevice.UnregisterForContainerSelectedEvent(this, this.containerTarget);

                // Unregisters for the DeviceUnpluggedEvent.  This will stop you from receiving notification when the device is unplugged.
                this.selectedDevice.UnregisterForDeviceUnpluggedEvent(this, this.unpluggedTarget);

                // Tells the device to stop reading while putting the device in DigitizerOff mode.
                this.selectedDevice.StopReading();

                this.selectedDevice.CurrentDisplay.ClearDisplay();

                // Closes the device
                this.selectedDevice.Close();
            }
        }

        public void RegisterForCoordinateReceivedEvent(object sender, CoordinateReceivedEventHandler target)
        {
            // Add the target as a CoordinateEvent handler
            this.CoordinateEvent += target;
        }

        public void UnregisterForCoordinateReceivedEvent(object sender, CoordinateReceivedEventHandler target)
        {
            // Remove the target as a CoordinateEvent handler
            this.CoordinateEvent -= target;
        }

        public void RegisterForSwipeReceivedEvent(object sender, SwipeReceivedEventHandler target)
        {
            // Add the target as a SwipeEvent handler
            this.SwipeEvent += target;
        }

        public void UnregisterForSwipeReceivedEvent(object sender, SwipeReceivedEventHandler target)
        {
            // Remove the target as a SwipeEvent handler
            this.SwipeEvent -= target;
        }

        public void SaveSignatureToDesktop()
        {
            this.sig.SaveAsImage(STSignature.ImageType.JPEG, 240, 64, Environment.GetFolderPath(Environment.SpecialFolder.Desktop) + "\\Signature.jpeg");
        }

        private void OnCardSwipe(object sender, STCardReader.CardSwipedEventArgs e)
        {
            SwipeReceivedEventArg arg = new SwipeReceivedEventArg(e);

            this.SwipeEvent(this, arg);
        }

        private void OnButtonSelected(object sender, STSeriesDevice.RegionSelectedEventArgs e)
        {
            // A new coordinate event arg
            CoordinateReceivedEventArg arg = new CoordinateReceivedEventArg(e.SigPoint, e.RegionNumber);

            // Throw the CoordinateEvent
            // this event is setup to call the drawing method in STSeriesExampleForm, OnCoordinateReceived
            this.CoordinateEvent(this, arg);

            // Add the new signature point to the signature
            this.sig.AddPoint(e.SigPoint);
        }

        private void OnContainerSelected(object sender, STSeriesDevice.RegionSelectedEventArgs e)
        {
            // A new coordinate event arg
            CoordinateReceivedEventArg arg = new CoordinateReceivedEventArg(e.SigPoint, e.RegionNumber);

            // Throw the CoordinateEvent
            // this event is setup to call the drawing method in STSeriesExampleForm, OnCoordinateReceived
            this.CoordinateEvent(this, arg);

            // Add the new signature point to the signature so that it can be saved later, if requested
            this.sig.AddPoint(e.SigPoint);
        }

        private void OnDeviceUnplugged(object sender, EventArgs e)
        {
            // Check to see if the device is opened
            if (this.DeviceIsOpened)
            {
                // Close the device
                this.selectedDevice.Close();
            }
        }

        // Event args used to send coordinate information to the form
        public class CoordinateReceivedEventArg : EventArgs
        {
            public CoordinateReceivedEventArg(STSignaturePoint signaturePoint, int regionNumber)
            {
                this.SigPoint = signaturePoint;
                this.RegionNumber = regionNumber;
            }
            public STSignaturePoint SigPoint { get; set; }
            public int RegionNumber { get; set; }
        }

        // Event args used to send swipe information to the form
        public class SwipeReceivedEventArg : EventArgs
        {
            public SwipeReceivedEventArg(STCardReader.CardSwipedEventArgs cardSwipe)
            {
                this.CardSwipe = cardSwipe;
            }
            public STCardReader.CardSwipedEventArgs CardSwipe { get; set; }
        }
    }
}