﻿// <copyright file="IdentificationCard.cs" company="Scriptel Corporation">
//      Copyright 2014 - Scriptel Corporation
// </copyright>
namespace EasyScriptAPI
{
    using System;
    using System.Text;

    /// <summary>
    /// This class represents an identification card such as a United States drivers
    /// license. <see href="http://www.aamva.org/?aspxerrorpath=/NR/rdonlyres/66260AD6-64B9-45E9-A253-B8AA32241BE0/0/2005DLIDCardSpecV2FINAL.pdf" />
    /// </summary>
    public class IdentificationCard
    {
        /// <summary>
        /// Track one of the identification card.
        /// </summary>
        private IdentificationCardTrackOne trackOne;

        /// <summary>
        /// Track two of the identification card.
        /// </summary>
        private IdentificationCardTrackTwo trackTwo;

        /// <summary>
        /// Track three of the identification card.
        /// </summary>
        private IdentificationCardTrackThree trackThree;

        /// <summary>
        /// This method attempts to parse identification card information out of a
        /// raw magnetic strip string.
        /// </summary>
        /// <param name="cardData">card data to parse</param>
        /// <returns>IdentificationCard data if parsed, null otherwise.</returns>
        public static IdentificationCard Parse(string cardData)
        {
            IdentificationCardTrackOne t1 = IdentificationCardTrackOne.Parse(cardData);
            IdentificationCardTrackTwo t2 = IdentificationCardTrackTwo.Parse(cardData);
            IdentificationCardTrackThree t3 = IdentificationCardTrackThree.Parse(cardData);
        
            if (t1 != null || t2 != null || t3 != null)
            {
                IdentificationCard card = new IdentificationCard();
                card.SetTrackOne(t1);
                card.SetTrackTwo(t2);
                card.SetTrackThree(t3);
                return card;
            }

            return null;
        }
    
        /// <summary>
        /// Returns a human readable string representing this object.
        /// </summary>
        /// <returns>Human readable string representing this object.</returns>
        public override string ToString()
        {
            StringBuilder sb = new StringBuilder();
            string ln = Environment.NewLine;
            sb.Append("Identification Card").Append(ln);
            if (this.trackOne != null)
            {
                sb.Append(this.trackOne).Append(ln);
            }

            if (this.trackTwo != null)
            {
                sb.Append(this.trackTwo).Append(ln);
            }

            if (this.trackThree != null)
            {
                sb.Append(this.trackThree).Append(ln);
            }

            return sb.ToString();
        }

        /// <summary>
        /// Gets the track one data for this card.
        /// </summary>
        /// <returns>Track one data for this card.</returns>
        public IdentificationCardTrackOne GetTrackOne()
        {
            return this.trackOne;
        }
        
        /// <summary>
        /// Gets the track two data for this card.
        /// </summary>
        /// <returns>Track two data for this card.</returns>
        public IdentificationCardTrackTwo GetTrackTwo()
        {
            return this.trackTwo;
        }

        /// <summary>
        /// Gets the track three data for this card.
        /// </summary>
        /// <returns>Track three data for this card.</returns>
        public IdentificationCardTrackThree GetTrackThree()
        {
            return this.trackThree;
        }

        /// <summary>
        /// Sets the track one data for this card.
        /// </summary>
        /// <param name="trackOne">Track one data for this card.</param>
        private void SetTrackOne(IdentificationCardTrackOne trackOne)
        {
            this.trackOne = trackOne;
        }

        /// <summary>
        /// Sets the track two data for this card.
        /// </summary>
        /// <param name="trackTwo">Track two data for this card.</param>
        private void SetTrackTwo(IdentificationCardTrackTwo trackTwo)
        {
            this.trackTwo = trackTwo;
        }

        /// <summary>
        /// Sets the track three data for this card.
        /// </summary>
        /// <param name="trackThree">Track three data for this card.</param>
        private void SetTrackThree(IdentificationCardTrackThree trackThree)
        {
            this.trackThree = trackThree;
        }
    }
}
