<?php
/**
 * This file contains the {@link IdentificationCardTrackTwo} class.
 * @copyright (c) 2015, Scriptel Corporation.
 */
namespace com\scriptel;

/**
 * This class represents track two of a magnetic strip identification card
 * such as a United States driver's license.
 */
class IdentificationCardTrackTwo {
    /**
     * Issuer identification number.
     * @var string
     */
    private $issuerNumber;
    /**
     * Document number.
     * @var string
     */
    private $idNumber;
    /**
     * Document Expiration date.
     * @var Date
     */
    private $expiration;
    /**
     * Card-holder Birth date.
     * @var Date
     */
    private $birthDate;

    /**
     * This method attempts to parse track two of an identification card.
     * @param string $trackData
     * @return \IdentificationCardTrackTwo
     */
    public static function parse($trackData) {
        $myPattern = ".;(6[0-9]{5})([0-9]{1,13})=([0-9]{4})([0-9]{8})([0-9]{0,5})=?\\?.";
        $matches[] = NULL;
        preg_match($myPattern, $trackData, $matches);
        if (count($matches) > 0 ) {
            $track = new IdentificationCardTrackTwo();
            $track->setIssuerNumber($matches[1]);
            $track->setIdNumber(trim($matches[2].$matches[5]));
            $expYear = 2000 + (substr($matches[3],0,2)*1);
            $expMonth = substr($matches[3],2)*1;
            $track->setExpiration(mktime(0,0,0,$expMonth+1,1,$expYear)-1);

            $birthYear = substr($matches[4],0,4)*1;
            $birthMonth = substr($matches[4],4,2)*1;
            $birthDay = substr($matches[4],6)*1;
            $track->setBirthDate(mktime(0,0,0,$birthMonth,$birthDay,$birthYear));

            return $track;
        }
        return NULL;
    }

    /**
     * Returns this object as a human readable string.
     * @return string
     */
    public function toString() {
        $strCardData = "";
        $strCardData .= "Identification Card - Track Two: \n";
        $strCardData .= "     Issuer Number: " . $this->getIssuerNumber() . "\n";
        $strCardData .= "         ID Number: " . $this->getIdNumber() . "\n";
        $strCardData .= "        Expiration: " . $this->getExpiration() . "\n";
        $strCardData .= "        Birth Date: " . $this->getBirthDate() . "\n";
        return $strCardData;
    }

    /**
     * Gets the issuer identification number. This can be used to determine
     * which state issued the document.
     * @return string
     */
    function getIssuerNumber() {
        return $this->issuerNumber;
    }
    /**
     * Gets the identification number of the document.
     * @return string
     */
    function getIdNumber() {
        return $this->idNumber;
    }
    /**
     * Gets the expiration date of the document.
     * @return Date
     */
    function getExpiration() {
        return $this->expiration;
    }
    /**
     * Gets the birth date of the card-holder.
     * @return Date
     */
    function getBirthDate() {
        return $this->birthDate;
    }
    /**
     * Sets the issuer identification number.
     * @param string $issuerNumber
     */
    function setIssuerNumber($issuerNumber) {
        $this->issuerNumber = $issuerNumber;
    }
    /**
     * Sets the identification number of the document.
     * @param string $idNumber
     */
    function setIdNumber($idNumber) {
        $this->idNumber = $idNumber;
    }
    /**
     * Sets the expiration date of the document.
     * @param Date $expiration
     */
    function setExpiration($expiration) {
        $this->expiration = $expiration;
    }
    /**
     * Sets the birth date of the card-holder.
     * @param Date $birthDate
     */
    function setBirthDate($birthDate) {
        $this->birthDate = $birthDate;
    }
}
