<?php
/**
 * This file contains the {@link FinancialCardTrackOne} class.
 * @copyright (c) 2015, Scriptel Corporation.
 */
namespace com\scriptel;

/**
 * This class represents track one of a financial card.
 */
class FinancialCardTrackOne {
    /**
     * The credit card number of this financial card.
     * @var string
     */
    private $accountNumber;
    /**
     * The first name on the card.
     * @var string
     */
    private $firstName;
    /**
     * The last name on the card.
     * @var string
     */
    private $lastName;
    /**
     * The expiration date on the card.
     * @var Date
     */
    private $expiration;
    /**
     * The service code on the card.
     * @var string
     */
    private $serviceCode;
    /**
     * Optional additional date the issuer has added on the strip.
     * @var string
     */
    private $discretionaryData;

    /**
     * This method attempts to parse track one of the raw track string.
     * @param string $trackData
     * @return FinancialCardTrackOne
     */
    public static function parse($trackData) {
        $myPattern = ".%[A-Z]{1}(\\d{1,19})\\^([^\\^]{1,30})\\^(\\d{2})(\\d{2})([0-9]{3})([A-Za-z 0-9]*)\\?.";
        $matches[] = NULL;
        preg_match($myPattern, $trackData, $matches);
        if (count($matches) > 0 ) {
            $track = new FinancialCardTrackOne();
            $track->setAccountNumber($matches[1]);
            $track->setServiceCode($matches[5]);
            $track->setDiscretionaryData($matches[6]);
            $name = $matches[2];
            $slashPosition = strpos($name, '/');
            if ($slashPosition >= 0) {
                $track->setLastName(trim(substr($name,0,$slashPosition)));
                $track->setFirstName(trim(substr($name,$slashPosition + 1)));
            } else {
                $track->setFirstName('');
                $track->setLastName(trim($name));
            }
            $expMonth = ($matches[4]*1);
            $expYear = ($matches[3]*1)+2000;
            $track->setExpiration(mktime(0,0,0,$expMonth+1,1,$expYear)-1);
            return $track;
        }
        return NULL;
    }
    /**
     * Returns this object as a human readable string.
     * @return string
     */
    public function toString() {
        $strCardData = "";
        $strCardData .= "Financial Card - Track One: \n";
        $strCardData .= "    Account Number: " . $this->getAccountNumber() . "\n";
        $strCardData .= "        First Name: " . $this->getFirstName() . "\n";
        $strCardData .= "         Last Name: " . $this->getLastName() . "\n";
        $strCardData .= "        Expiration: " . $this->getExpiration() . "\n";
        $strCardData .= "      Service Code: " . $this->getServiceCode() . "\n";
        $strCardData .= "Discretionary Data: " . $this->getDiscretionaryData() . "\n";
        return $strCardData;
    }
    /**
     * Gets the account number on the card.
     * @return string.
     */
    function getAccountNumber() {
        return $this->accountNumber;
    }
    /**
     * Gets the first name on the card.
     * @return string.
     */
    function getFirstName() {
        return $this->firstName;
    }
    /**
     * Gets the last name on the card.
     * @return string
     */
    function getLastName() {
        return $this->lastName;
    }
    /**
     * Gets the expiration date on the card.
     * @return Date
     */
    function getExpiration() {
        return $this->expiration;
    }
    /**
     * Gets the service code on the card.
     * @return string
     */
    function getServiceCode() {
        return $this->serviceCode;
    }
    /**
     * Gets the vendor-specific discretionary data on the card.
     * @return string
     */
    function getDiscretionaryData() {
        return $this->discretionaryData;
    }
    /**
     * Sets the account number on the card.
     * @param string $cardNumber
     */
    function setAccountNumber($cardNumber) {
        $this->accountNumber = $cardNumber;
    }
    /**
     * Sets the first name on the card.
     * @param string $firstName
     */
    function setFirstName($firstName) {
        $this->firstName = $firstName;
    }
    /**
     * Sets the last name on the card.
     * @param string $lastName
     */
    function setLastName($lastName) {
        $this->lastName = $lastName;
    }
    /**
     * Sets the expiration date on the card.
     * @param Date $expiration
     */
    function setExpiration($expiration) {
        $this->expiration = $expiration;
    }
    /**
     * Sets the service code on the card.
     * @param string $serviceCode
     */
    function setServiceCode($serviceCode) {
        $this->serviceCode = $serviceCode;
    }
    /**
     * Sets the vendor-specific discretionary data.
     * @param string $discretionaryData
     */
    function setDiscretionaryData($discretionaryData) {
        $this->discretionaryData = $discretionaryData;
    }
}
