/*
 * Copyright 2013 - Scriptel Corporation
 */
package com.scriptel.easyscript;

/**
 * This class represents an identification card such as a United States drivers
 * license.
 * @see <a href="http://www.aamva.org/?aspxerrorpath=/NR/rdonlyres/66260AD6-64B9-45E9-A253-B8AA32241BE0/0/2005DLIDCardSpecV2FINAL.pdf">http://www.aamva.org/?aspxerrorpath=/NR/rdonlyres/66260AD6-64B9-45E9-A253-B8AA32241BE0/0/2005DLIDCardSpecV2FINAL.pdf</a>
 */
public class IdentificationCard {
    private IdentificationCardTrackOne trackOne;
    private IdentificationCardTrackTwo trackTwo;
    private IdentificationCardTrackThree trackThree;

    /**
     * This method attempts to parse identification card information out of a
     * raw magnetic strip string.
     * @param cardData Card data to parse.
     * @return Identification Card data if parsed, null otherwise.
     */
    public static IdentificationCard parse(String cardData) {
        IdentificationCardTrackOne t1 = IdentificationCardTrackOne.parse(cardData);
        IdentificationCardTrackTwo t2 = IdentificationCardTrackTwo.parse(cardData);
        IdentificationCardTrackThree t3 = IdentificationCardTrackThree.parse(cardData);
        
        if(t1!=null || t2!=null || t3!=null) {
            IdentificationCard card = new IdentificationCard();
            card.setTrackOne(t1);
            card.setTrackTwo(t2);
            card.setTrackThree(t3);
            return card;
        }
        return null;
    }
    
    /**
     * Returns this object as a human readable string.
     * @return String containing human readable string.
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        String ln = System.getProperty("line.separator");
        sb.append("Identification Card").append(ln);
        if(trackOne!=null) {
            sb.append(trackOne).append(ln);
        }
        if(trackTwo!=null) {
            sb.append(trackTwo).append(ln);
        }
        if(trackThree!=null) {
            sb.append(trackThree).append(ln);
        }
        
        return sb.toString();
    }
    /**
     * Gets parsed track one data.
     * @return Track one data.
     */
    public IdentificationCardTrackOne getTrackOne() {
        return trackOne;
    }

    /**
     * Sets parsed track one data.
     * @param trackOne the trackOne to set
     */
    private void setTrackOne(IdentificationCardTrackOne trackOne) {
        this.trackOne = trackOne;
    }

    /**
     * Gets parsed track two data.
     * @return parsed track two data.
     */
    public IdentificationCardTrackTwo getTrackTwo() {
        return trackTwo;
    }

    /**
     * Sets track two data.
     * @param trackTwo the trackTwo to set
     */
    private void setTrackTwo(IdentificationCardTrackTwo trackTwo) {
        this.trackTwo = trackTwo;
    }

    /**
     * Gets track three data.
     * @return Parsed track three data.
     */
    public IdentificationCardTrackThree getTrackThree() {
        return trackThree;
    }

    /**
     * Sets track three data.
     * @param trackThree the trackThree to set
     */
    private void setTrackThree(IdentificationCardTrackThree trackThree) {
        this.trackThree = trackThree;
    }
}
