﻿// <copyright file="CardSwipe.cs" company="Scriptel Corporation">
//      Copyright 2014 - Scriptel Corporation
// </copyright>
namespace EasyScriptAPI
{
    using System;
    using System.Text;

    /// <summary>
    /// This class represents a successful magnetic card swipe from a ScripTouch
    /// EasyScript device with a magnetic card strip reader.
    /// </summary>
    public class CardSwipe
    {
        /// <summary>
        ///  The protocol version being used to decode the string.
        /// </summary>
        private string protocolVersion;

        /// <summary>
        /// The raw card data.
        /// </summary>
        private string cardData;

        /// <summary>
        /// If the card is a recognizable financial card, this will contain parsed
        /// financial card data.
        /// </summary>
        private FinancialCard financialCard;

        /// <summary>
        /// If the card is a recognizable identification card this will contain
        /// parsed identification data.
        /// </summary>
        private IdentificationCard identificationCard;

        /// <summary>
        /// Returns a human readable string representation of this object.
        /// </summary>
        /// <returns>Human readable string representation of this object.</returns>
        public override string ToString()
        {
            StringBuilder sb = new StringBuilder();
            string ln = Environment.NewLine;
            sb.Append("Scriptel EasyScript Card Swipe").Append(ln);
            sb.Append("Protocol Version: ").Append(this.protocolVersion).Append(ln);
            sb.Append("       Card Data: ").Append(this.cardData).Append(ln);
            if (this.financialCard != null)
            {
                sb.Append(this.financialCard);
            }

            if (this.identificationCard != null)
            {
                sb.Append(this.identificationCard);
            }
        
            return sb.ToString();
        }
    
        /// <summary>
        /// Returns the protocol version
        /// </summary>
        /// <returns>the protocol version</returns>
        public string GetProtocolVersion()
        {
            return this.protocolVersion;
        }

        /// <summary>
        /// Gets the raw card swipe data
        /// </summary>
        /// <returns>the raw card swipe data</returns>
        public string GetCardData()
        {
            return this.cardData;
        }

        /// <summary>
        /// Gets parsed financial card data, null if not set.
        /// </summary>
        /// <returns>financial card data</returns>
        public FinancialCard GetFinancialCard()
        {
            return this.financialCard;
        }

        /// <summary>
        /// Gets the identification card data
        /// </summary>
        /// <returns>the identification card data</returns>
        public IdentificationCard GetIdentificationCard()
        {
            return this.identificationCard;
        }

        /// <summary>
        /// Sets parsed financial card data
        /// </summary>
        /// <param name="financialCard">the financial card data</param>
        internal void SetFinancialCard(FinancialCard financialCard)
        {
            this.financialCard = financialCard;
        }

        /// <summary>
        /// Sets the raw card swipe data
        /// </summary>
        /// <param name="cardData">The raw card swipe data</param>
        internal void SetCardData(string cardData)
        {
            this.cardData = cardData;
        }

        /// <summary>
        /// sets the identification card data
        /// </summary>
        /// <param name="identificationCard">the identification card data</param>
        internal void SetIdentificationCard(IdentificationCard identificationCard)
        {
            this.identificationCard = identificationCard;
        }

        /// <summary>
        /// Sets the protocol version
        /// </summary>
        /// <param name="protocolVersion">The protocol version</param>
        internal void SetProtocolVersion(string protocolVersion)
        {
            this.protocolVersion = protocolVersion;
        }
    }
}
