﻿// <copyright file="BinaryTreeReader.cs" company="Scriptel Corporation">
//      Copyright 2015 - Scriptel Corporation
// </copyright>
namespace EasyScriptAPI
{
    using System;

    class BinaryTreeReader
    {
        /// <summary>
        /// The current location in the binary tree.
        /// </summary>
        private int location = 0;
        /// <summary>
        /// The current bit depth.
        /// </summary>
        private int bitDepth = 1;
        /// <summary>
        /// Static huffman table used to decode signature points.
        /// </summary>
        private string table =
            "0011000000100000011101101000000001011100000000001010000010000000001000000000" +
            "1000100000001100010000000100010010000000111010010000001010110010001110000000" +
            "1011101000000100101000000000010010010000001001110000000010010000000001010100" +
            "0000001000101000000100000001111010000000000110001000000010100100000011110000" +
            "1000000101010100000001011110000000100110100000000011101000000001110100000000" +
            "1100000000100000011001110000001101110100000010001101000000111010100000011001" +
            "0001000000011011100000001110010000000010100100000011110101000011000000010000" +
            "0010011001000000100111001000110000000000100111100000001101001000000101111000" +
            "0000101000100000001010000000000100101100000010000001011010010000000000000010" +
            "0000010010101000000101110100000011000110000000100000010000000011011000000010" +
            "1100100000000010110000000100100000100000000011001000000010101100000010100001" +
            "0000000010001000000001111000001000000110110000010011000000001001101000000001" +
            "0101010000001011001000000011001100000001100010000000001010010000000101101000" +
            "0000010101110000001011011000000000010111110000001101000000000011000000000001" +
            "1011000000000011000010000001101101000000011001010000001010110000000000110001" +
            "1000000101100000000001101010000000110011100000000110010000000010100110000000" +
            "1101011000000101110000000010001000000000100000010001011011100000001000000000" +
            "0011000000111110000100000000100101000000011111010000001010011000000101111100" +
            "0010000000000010000001110010100000011010010000001110001000000101100010000001" +
            "1111110000001000010001000000110101100000010000010000000110100010000001100001" +
            "0000001001001000000011001";

        /// <summary>
        /// Private constructor to prevent this class from being instantiated.
        /// </summary>
        private BinaryTreeReader()
        {
        }

        /// <summary>
        /// Static method for building the binary tree.
        /// </summary>
        /// <returns>Binary tree representing the decompression huffman table.</returns>
        public static BinaryTree GetCompressionTree()
        {
            BinaryTreeReader reader = new BinaryTreeReader();
            reader.location = 0;
            reader.bitDepth = reader.ReadTableValue(6);
            return reader.WriteTree();
        }

        /// <summary>
        /// Gets a value from the huffman table.
        /// </summary>
        /// <param name="depth">Depth of the value to read.</param>
        /// <returns>Value read from the table.</returns>
        public int ReadTableValue(int depth)
        {
            int v = 0;
            for (int i = 0; i < depth; i++)
            {
                char c = table[location++];
                v = (v * 2) + ((c == '0') ? 0 : 1);
            }
            return v;
        }

        /// <summary>
        /// Reads the next node from the huffman table and writes it to the current
        /// binary tree.
        /// </summary>
        /// <returns>Binary tree representing the values read from the table.</returns>
        public BinaryTree WriteTree()
        {
            BinaryTree node = new BinaryTree();
            int type = ReadTableValue(1);
            if (type == 0)
            {
                node.Left = WriteTree();
                node.Right = WriteTree();
            }
            else
            {
                node.Value = ReadTableValue(bitDepth);
            }
            return node;
        }

    }

    /// <summary>
    /// Binary tree class representing a leaf or branch in the huffman table.
    /// </summary>
    class BinaryTree
    {
        /// <summary>
        /// If this node is a leaf this value wil be non-null and contain the
        /// value of the leaf.
        /// </summary>
        public int? Value
        {
            get;
            set;
        }

        /// <summary>
        /// If this node is a branch this will contain the reference to the
        /// left of the branch.
        /// </summary>
        public BinaryTree Left
        {
            get;
            set;
        }

        /// <summary>
        /// If this node is a branch this will contain the reference to the
        /// right of the branch.
        /// </summary>
        public BinaryTree Right
        {
            get;
            set;
        }
    }
}
