/*
 * Copyright 2013 - Scriptel Corporation
 */
package com.scriptel.easyscript.example;

import com.scriptel.easyscript.CardSwipe;
import com.scriptel.easyscript.CardSwipeInvalidException;
import com.scriptel.easyscript.EasyScript;
import com.scriptel.easyscript.Signature;
import com.scriptel.easyscript.SignatureInvalidException;
import java.awt.Canvas;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.KeyEventDispatcher;
import java.awt.KeyboardFocusManager;
import java.awt.event.KeyEvent;
import java.awt.image.BufferedImage;
import javax.swing.JTextArea;

/**
 * A simple application demonstrating the use of the Scriptel ScripTouch
 * EasyScript Java API in a SWING application.
 */
public class LegacyExample {
    private ExampleForm form;
    private StringBuilder buffer = new StringBuilder();
    private EasyScript eScript;
    private boolean signature;
    private boolean capturing = false;
    
    public LegacyExample() {
        //Initialize and show our form.
        form = new ExampleForm();
        form.setVisible(true);
        
        //Create a new EasyScript Object.
        eScript = new EasyScript();
        
        //Create an event listener looking for key presses on our form.
        KeyboardFocusManager mgr = KeyboardFocusManager.getCurrentKeyboardFocusManager();
        mgr.addKeyEventDispatcher(new ScriptelKeyboardDispatcher());
    }
    
    public static void main(String[] args) {
        LegacyExample app = new LegacyExample();
    }
    
    /**
     * This class listens for keyboard events and handles them as they occur.
     */
    private class ScriptelKeyboardDispatcher implements KeyEventDispatcher {
        @Override
        public boolean dispatchKeyEvent(KeyEvent ke) {
            //Check to make sure this key was pressed.
            if(ke.getID() == KeyEvent.KEY_TYPED) {
                //Check to see if the start character is the start character of either the signature or card swipe protocol.
                if(!capturing && ke.getKeyChar()==eScript.getSignatureProtocol().getStartStream()) {
                    capturing = true;
                    signature = true;
                } else if(!capturing && ke.getKeyChar()==eScript.getCardProtocol().getStartStream()) {
                    capturing = true;
                    signature = false;
                }
                
                //If we're capturing append the character to our buffer.
                if(capturing) {
                    buffer.append(ke.getKeyChar());
                    //this prevents the key event from being passed to other form elements, such as a JTextArea
                    ke.consume();
                }
                
                //Check to see if the character is an ending character for signatures or card swipes.
                //If it is, pass off the string to the handlers.
                if(capturing && signature && ke.getKeyChar()==eScript.getSignatureProtocol().getEndStream()) {
                    capturing = false;
                    try {
                        Signature s = eScript.parseSignature(buffer.toString());
                        handleSignature(s);
                    } catch(SignatureInvalidException e) {
                        handleException(e);
                    }
                    buffer.delete(0, buffer.length());
                } else if(capturing && !signature && ke.getKeyChar()==eScript.getCardProtocol().getEndStream()) {
                    capturing = false;
                    try {
                        CardSwipe s = eScript.parseCardSwipe(buffer.toString());
                        handleCardSwipe(s);
                    } catch(CardSwipeInvalidException e) {
                        handleException(e);
                    }
                    buffer.delete(0, buffer.length());
                }
            }
            return false;
        }
    }
    /**
     * This method handles incoming signatures.
     * @param s Signature object received from EasyScript device.
     */
    private void handleSignature(Signature s) {
        Canvas canvas = form.getCanvas();
        JTextArea textArea = form.getTextArea();
        
        BufferedImage img = eScript.renderSignature(s, Color.black, Color.white, 2, 2);
        canvas.getGraphics().drawImage(img, 0, 0, null);
        textArea.setText(s.toString());
        textArea.setSelectionStart(0);
        textArea.setSelectionEnd(0);
    }
    /**
     * This method handles incoming card swipes.
     * @param s Card Swipe object received from the EasyScript device.
     */
    private void handleCardSwipe(CardSwipe s) {
        Canvas canvas = form.getCanvas();
        JTextArea textArea = form.getTextArea();
        
        Graphics2D ctx = (Graphics2D)canvas.getGraphics();
        ctx.setColor(Color.white);
        ctx.fillRect(0, 0, 480, 128);
        
        textArea.setText(s.toString());
        textArea.setSelectionStart(0);
        textArea.setSelectionEnd(0);
    }
    /**
     * This method handles exceptions.
     * @param s Exception to handle.
     */
    private void handleException(Exception s) {
        Canvas canvas = form.getCanvas();
        JTextArea textArea = form.getTextArea();
        
        Graphics2D ctx = (Graphics2D)canvas.getGraphics();
        ctx.setColor(Color.white);
        ctx.fillRect(0, 0, 480, 128);
        
        textArea.setText(s.toString()+"\n"+buffer);
        textArea.setSelectionStart(0);
        textArea.setSelectionEnd(0);
    }
}
