/*
 * Copyright 2013 - Scriptel Corporation
 */
package com.scriptel.easyscript;
import java.util.List;
/**
 * This class represents a parsed signature from a Scriptel ScripTouch EasyScript
 * device.
 */
public class Signature {
    private final String protocolVersion;
    private final String model;
    private final String version;
    private final List<List<Coordinate>> strokes;
    
    /**
     * Constructor, creates a new instance of this type.
     * @param metaData all of the non-signature related data.
     * @param strokes List of strokes containing coordinates.
     */
    public Signature(SignatureMetaData metaData, List<List<Coordinate>> strokes) {
        this.protocolVersion = metaData.getProtocolVersion();
        this.model = metaData.getModel();
        this.version = metaData.getVersion();
        this.strokes = strokes;
    }

    /**
     * Produces a human readable representation of this object.
     * @return Human readable representation of this object.
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("Scriptel EasyScript Signature:\n");
        sb.append("   Protocol Version: ").append(protocolVersion).append("\n");
        sb.append("       Device Model: ").append(model).append("\n");
        sb.append("   Firmware Version: ").append(version).append("\n");
        sb.append("            Strokes: \n");
        for(int x=0;x<this.strokes.size();x++) {
            sb.append("   Stroke ").append(x).append(":\n");
            for(Coordinate c : strokes.get(x)) {
                sb.append("      ").append(c).append("\n");
            }
            sb.append("\n");
        }
        return sb.toString();
    }
    
    /**
     * Gets the protocol version used to generate the EasyScript string.
     * @return Protocol version.
     */
    public String getProtocolVersion() {
        return protocolVersion;
    }

    /**
     * The model number of the device used to create the string.
     * @return Model number
     */
    public String getModel() {
        return model;
    }

    /**
     * The firmware version of the device used to create the string.
     * @return Firmware version.
     */
    public String getVersion() {
        return version;
    }
    
    /**
     * Gets the metadata associated with this signature.
     * @return Metadata associated with this signature.
     */
    public SignatureMetaData getMetaData() {
        return new SignatureMetaData(getProtocolVersion(), getModel(), getVersion());
    }

    /**
     * List of strokes containing lists of points that make up the encoded
     * signature.
     * @return List of strokes.
     */
    public List<List<Coordinate>> getStrokes() {
        return strokes;
    }
}
