/*
 * Copyright 2013 - Scriptel Corporation
 */
package com.scriptel.easyscript;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * This class represents track one of the magnetic strip of an identification
 * card, such as a United States drivers license.
 */
public class IdentificationCardTrackOne {
    /**
     * Card-holder State
     */
    private String state;
    /**
     * Card-holder City
     */
    private String city;
    /**
     * Card-holder last name.
     */
    private String lastName;
    /**
     * Card-holder first name.
     */
    private String firstName;
    /**
     * Card-holder middle name.
     */
    private String middleName;
    /**
     * Card-holder address.
     */
    private String address;
    /**
     * Vendor-specific discretionary data.
     */
    private String discretionaryData;

    /**
     * This method attempts to parse the first track of data out of a magnetic
     * strip string.
     * @param trackData Data to parse from.
     * @return IdentificationCardTrackOne if data can be parsed, null otherwise.
     */
    public static IdentificationCardTrackOne parse(String trackData) {
        Pattern p = Pattern.compile(".*\\%([A-Z]{2})([A-Z\\.\\-' ]{1,13})\\^?([A-Z\\.\\-' \\$]{1,35})\\^?([^\\?^\\^]{1,29})\\^?\\?.*");
        Matcher m = p.matcher(trackData);
        if(m.matches()) {
            IdentificationCardTrackOne track = new IdentificationCardTrackOne();
            track.setState(m.group(1));
            track.setCity(m.group(2));
            track.setAddress(m.group(4));
            String[] nameBits = m.group(3).split("\\$");
            if(nameBits.length>0) {
                track.setLastName(nameBits[0]);
            }
            if(nameBits.length>1) {
                track.setFirstName(nameBits[1]);
            }
            if(nameBits.length>2) {
                track.setMiddleName(nameBits[2]);
            }
            return track;
        }
        
        return null;
    }
    
    /**
     * Returns this object as a human readable string.
     * @return String containing human readable string.
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        String ln = System.getProperty("line.separator");
        sb.append("Identification Card - Track One:").append(ln);
        sb.append("             State: ").append(state).append(ln);
        sb.append("              City: ").append(city).append(ln);
        sb.append("        First Name: ").append(firstName).append(ln);
        sb.append("       Middle Name: ").append(middleName).append(ln);
        sb.append("         Last Name: ").append(lastName).append(ln);
        sb.append("        Expiration: ").append(address).append(ln);
        sb.append("Discretionary Data: ").append(discretionaryData).append(ln);
        return sb.toString();
    }
    
    /**
     * Gets the card-holders state.
     * @return the state
     */
    public String getState() {
        return state;
    }

    /**
     * Sets the card-holders state.
     * @param state the state to set
     */
    private void setState(String state) {
        this.state = state;
    }

    /**
     * Gets the card-holders city.
     * @return the city
     */
    public String getCity() {
        return city;
    }

    /**
     * Sets the card-holders city.
     * @param city the city to set
     */
    private void setCity(String city) {
        this.city = city;
    }

    /**
     * Gets the last name of the card-holder.
     * @return the lastName
     */
    public String getLastName() {
        return lastName;
    }

    /**
     * Sets the last name of the card-holder.
     * @param lastName the lastName to set
     */
    private void setLastName(String lastName) {
        this.lastName = lastName;
    }

    /**
     * Gets the first name of the card-holder.
     * @return the firstName
     */
    public String getFirstName() {
        return firstName;
    }

    /**
     * Sets the first name of the card-holder.
     * @param firstName the firstName to set
     */
    private void setFirstName(String firstName) {
        this.firstName = firstName;
    }

    /**
     * Gets the middle name of the card-holder.
     * @return the middleName
     */
    public String getMiddleName() {
        return middleName;
    }

    /**
     * Sets the middle name of the card holder.
     * @param middleName the middleName to set
     */
    private void setMiddleName(String middleName) {
        this.middleName = middleName;
    }

    /**
     * Gets the address of the card-holder.
     * @return Address of card holder.
     */
    public String getAddress() {
        return address;
    }

    /**
     * Sets the address of the card-holder.
     * @param address Address of the card-holder.
     */
    private void setAddress(String address) {
        this.address = address;
    }

    /**
     * Gets vendor-specific discretionary data.
     * @return the discretionaryData
     */
    public String getDiscretionaryData() {
        return discretionaryData;
    }

    /**
     * Sets vendor-specific discretionary data.
     * @param discretionaryData the discretionaryData to set
     */
    private void setDiscretionaryData(String discretionaryData) {
        this.discretionaryData = discretionaryData;
    }
}
