﻿// <copyright file="TestEasyScript.cs" company="Scriptel Corporation">
//      Copyright 2014 - Scriptel Corporation
// </copyright>
namespace TestEasyScriptAPI
{
    using System;
    using System.Collections;
    using System.Collections.Generic;
    using EasyScriptAPI;
    using Microsoft.VisualStudio.TestTools.UnitTesting;

    /// <summary>
    /// This test class is used to test the EasyScript API.
    /// </summary>
    [TestClass]
    public class TestEasyScript
    {
        private class HighResNativeProtocol : STNSignatureProtocol {
            public override int GetHeight() {
                return 2999;
            }

            public override int GetWidth() {
                return 2999;
            }
        }

        private class LowResNativeProtocol : STNSignatureProtocol {
            public override int GetHeight() {
                return 505;
            }

            public override int GetWidth() {
                return 505;
            }
        }

        private EasyScript eScriptLR;
        private EasyScript eScriptHR;

        [TestInitialize()]
        public void Initialize()
        {
            HighResNativeProtocol highResProtocol = new HighResNativeProtocol();
            eScriptHR = new EasyScript(highResProtocol, new STNCardSwipeProtocol());

            LowResNativeProtocol lowResProtocol = new LowResNativeProtocol();
            eScriptLR = new EasyScript(lowResProtocol, new STNCardSwipeProtocol());
        }

        /// <summary>
        /// This method tests parsing of signatures.
        /// </summary>
        [TestMethod]
        public void TestParseSignature()
        {
            Signature signature = eScriptLR.ParseSignature("~STSIGN A ST1526-STN 03.07.86 OLsnOB90 OJsfO@s2O_u8O)w;OVy\\OJ1hOF1fOH1dOR1fO)1vQD32QD7fO*9lOD9t `");
            Assert.AreEqual(2, signature.GetStrokes().Count);
            Assert.AreEqual(2, signature.GetStrokes()[0].Count);

            Assert.AreEqual(166.0, signature.GetStrokes()[0][0].X);
            Assert.AreEqual(213.0, signature.GetStrokes()[0][0].Y);
            Assert.AreEqual(161.0, signature.GetStrokes()[0][1].X);
            Assert.AreEqual(408.0, signature.GetStrokes()[0][1].Y);

            Assert.AreEqual(14, signature.GetStrokes()[1].Count);
            Assert.AreEqual(165.0, signature.GetStrokes()[1][0].X);
            Assert.AreEqual(209.0, signature.GetStrokes()[1][0].Y);
            Assert.AreEqual(174.0, signature.GetStrokes()[1][1].X);
            Assert.AreEqual(220.0, signature.GetStrokes()[1][1].Y);
            Assert.AreEqual(179.0, signature.GetStrokes()[1][2].X);
            Assert.AreEqual(246.0, signature.GetStrokes()[1][2].Y);
            Assert.AreEqual(178.0, signature.GetStrokes()[1][3].X);
            Assert.AreEqual(274.0, signature.GetStrokes()[1][3].Y);
            Assert.AreEqual(171.0, signature.GetStrokes()[1][4].X);
            Assert.AreEqual(296.0, signature.GetStrokes()[1][4].Y);
            Assert.AreEqual(165.0, signature.GetStrokes()[1][5].X);
            Assert.AreEqual(302.0, signature.GetStrokes()[1][5].Y);
            Assert.AreEqual(163.0, signature.GetStrokes()[1][6].X);
            Assert.AreEqual(301.0, signature.GetStrokes()[1][6].Y);
            Assert.AreEqual(164.0, signature.GetStrokes()[1][7].X);
            Assert.AreEqual(300.0, signature.GetStrokes()[1][7].Y);
            Assert.AreEqual(169.0, signature.GetStrokes()[1][8].X);
            Assert.AreEqual(301.0, signature.GetStrokes()[1][8].Y);
            Assert.AreEqual(178.0, signature.GetStrokes()[1][9].X);
            Assert.AreEqual(309.0, signature.GetStrokes()[1][9].Y);
            Assert.AreEqual(185.0, signature.GetStrokes()[1][10].X);
            Assert.AreEqual(335.0, signature.GetStrokes()[1][10].Y);
            Assert.AreEqual(185.0, signature.GetStrokes()[1][11].X);
            Assert.AreEqual(370.0, signature.GetStrokes()[1][11].Y);
            Assert.AreEqual(177.0, signature.GetStrokes()[1][12].X);
            Assert.AreEqual(396.0, signature.GetStrokes()[1][12].Y);
            Assert.AreEqual(162.0, signature.GetStrokes()[1][13].X);
            Assert.AreEqual(400.0, signature.GetStrokes()[1][13].Y);
        }

        /// <summary>
        /// This method tests parsing of 2 signatures in a row
        /// </summary>
        [TestMethod]
        public void TestParseTwoSignatures()
        {
            for (int x = 0; x < 2; x++)
            {
                Signature signature = eScriptLR.ParseSignature("~STSIGN A ST1526-STN 03.07.86 OLsnOB90 OJsfOD9t `");
                Assert.AreEqual(2, signature.GetStrokes().Count);
                Assert.AreEqual(2, signature.GetStrokes()[0].Count);

                Assert.AreEqual(166.0, signature.GetStrokes()[0][0].X);
                Assert.AreEqual(213.0, signature.GetStrokes()[0][0].Y);
                Assert.AreEqual(161.0, signature.GetStrokes()[0][1].X);
                Assert.AreEqual(408.0, signature.GetStrokes()[0][1].Y);

                Assert.AreEqual(2, signature.GetStrokes()[1].Count);
                Assert.AreEqual(165.0, signature.GetStrokes()[1][0].X);
                Assert.AreEqual(209.0, signature.GetStrokes()[1][0].Y);
                Assert.AreEqual(162.0, signature.GetStrokes()[1][1].X);
                Assert.AreEqual(400.0, signature.GetStrokes()[1][1].Y);
            }
        }

        /// <summary
        /// Test if the signature is empty; should throw exception
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(SignatureInvalidException))]
        public void TestUncompressedSignatureInBatchModeEmpty() {
            eScriptLR.ParseSignature("");
        }

        /// <summary>
        /// Test if the signature is null; should throw exception
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(SignatureInvalidException))]
        public void TestUncompressedSignatureInBatchModeNull() {
            eScriptHR.ParseSignature(null);
        }

        /// <summary>
        /// Test if there is no end of signature character (`); should throw exception
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(SignatureInvalidException))]
        public void TestUncompressedSignatureInBatchModeNoEndOfSignatureSymbol() {
            eScriptLR.ParseSignature("~STSIGN B ST1526-STN 03.07.63 AJah ");
        }

        /// <summary>
        /// Test if there is no start of signature character (~); should throw exception
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(SignatureInvalidException))]
        public void TestUncompressedSignatureInBatchModeNoStartOfSignatureSymbol() {
            eScriptLR.ParseSignature("STSIGN B ST1526-STN 03.07.63 AJah `");
        }

        /// <summary>
        /// Test if the sentinel is incorrect; should throw exception
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(SignatureInvalidException))]
        public void TestUncompressedSignatureInBatchModeWrongSentinel()  {
            eScriptLR.ParseSignature("~STSIGS B ST1526-STN 03.07.63 AJah `");
        }

        /// <summary>
        /// Test if X part has un-shifted chars like j instead of J in position 2 in the string; should throw exception
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(SignatureInvalidException))]
        public void TestUncompressedSignatureUnshiftedX() {
            eScriptLR.ParseSignature("~STSIGN B ST1526-STN 03.07.63 AJahAJAn `");
        }

        /// <summary>
        /// Test if Y part has shifted chars like A instead of a in position 3 in the string; should throw exception
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(SignatureInvalidException))]
        public void TestUncompressedSignatureShiftedY() {
            eScriptLR.ParseSignature("~STSIGN B ST1526-STN 03.07.63 AJahAJAn `");
        }


        /// <summary>
        /// Test to see if an incomplete signature throws an exception; should throw exception
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(SignatureInvalidException))]
        public void testUncompressedSignatureNumberOfTokens2() {
            eScriptLR.ParseSignature("~STSIGN B ST1526-STN 03.07.63");
        }    

        class BatchListener : IEasyScriptEventListener {
            private readonly List<List<Coordinate>> strokes = new List<List<Coordinate>>();
            private List<Coordinate> stroke;
            private CardSwipe swipe;
            private bool done = false;
            private bool canceled = false;
            SignatureMetaData metadata;

            
            public List<List<Coordinate>> GetStrokes() {
                return strokes;
            }
            
            public CardSwipe GetSwipe() {
                return swipe;
            }

            public bool IsDone() {
                return done;
            }

            public bool IsCanceled()
            {
                return canceled;
            }

            public SignatureMetaData GetMetadata() {
                return metadata;
            }

            public void ReceiveCoordinate(Coordinate coordinate) {
                stroke.Add(coordinate);
            }

            public void NewStroke() {
                stroke = new List<Coordinate>();
                strokes.Add(stroke);
            }

            public void Cancel() {
                canceled = true;
                done = true;
            }

            public void EndOfSignature() {
                done = true;
            }

            public void SignatureMetaData(SignatureMetaData md) {
                metadata = md;
            }

            public void CardSwipe(CardSwipe swipe) {
                this.swipe = swipe;
            }
        }

        /// <summary>
        /// Tests a signature with a card swipe happening in the middle of it.
        /// </summary>
        [TestMethod]
        public void TestCardSwipeWithinSignature()
        {
        
            String signatureSwipe = "~STSIGN C ST1526-STN 03.07.80 W@s;W@s;W@s.!STCARD A %B6131380999977777^SMITH/STEVE J^171256789012?;1234567890123456=17125678901234567890?\rWXuhWPu2WLu[WFwl U:s\\U:s\\U:s\\WBubWFutWHu6WLwf `";

            BatchListener listener = new BatchListener();
            eScriptLR.AddListener(listener);
            for (int i = 0; i < signatureSwipe.Length; i++)
            {
                eScriptLR.Parse(signatureSwipe[i]);
            }
        
            CardSwipe cs = listener.GetSwipe();

            IdentificationCard ic = cs.GetIdentificationCard();
            FinancialCard fc = cs.GetFinancialCard();
        
            Assert.IsNull(ic);
        
            Assert.IsNotNull(fc);
            Assert.AreEqual("6131380999977777", fc.GetTrackOne().GetAccountNumber());
            Assert.AreEqual("STEVE J", fc.GetTrackOne().GetFirstName());
            Assert.AreEqual("SMITH", fc.GetTrackOne().GetLastName());
            DateTime expDate = fc.GetTrackOne().GetExpiration();

            Assert.AreEqual(2017, expDate.Year);
            Assert.AreEqual(12, expDate.Month);

            Assert.AreEqual("1234567890123456", fc.GetTrackTwo().GetAccountNumber());
        
            expDate = fc.GetTrackTwo().GetExpiration();

            Assert.AreEqual(12, expDate.Month);
        
            Assert.IsTrue(listener.IsDone());
            SignatureMetaData metadata = listener.GetMetadata();
            Assert.AreEqual("C", metadata.ProtocolVersion);
            Assert.AreEqual("ST1526-STN", metadata.Model);
            Assert.AreEqual("03.07.80", metadata.Version);
        
            List<List<Coordinate>> strokes = listener.GetStrokes();
            Assert.AreEqual(2, strokes.Count);
            Assert.AreEqual(7, strokes[0].Count);

            Assert.AreEqual(251.0, strokes[1][0].X);
            Assert.AreEqual(227.0, strokes[1][0].Y);
            Assert.AreEqual(251.0, strokes[1][1].X);
            Assert.AreEqual(227.0, strokes[1][1].Y);
            Assert.AreEqual(251.0, strokes[1][2].X);
            Assert.AreEqual(227.0, strokes[1][2].Y);
            Assert.AreEqual(253.0, strokes[1][3].X);
            Assert.AreEqual(230.0, strokes[1][3].Y);
            Assert.AreEqual(255.0, strokes[1][4].X);
            Assert.AreEqual(239.0, strokes[1][4].Y);
            Assert.AreEqual(256.0, strokes[1][5].X);
            Assert.AreEqual(245.0, strokes[1][5].Y);
            Assert.AreEqual(258.0, strokes[1][6].X);
            Assert.AreEqual(255.0, strokes[1][6].Y);

            Assert.AreEqual(7, strokes[1].Count);

            Assert.AreEqual(251.0, strokes[1][0].X);
            Assert.AreEqual(227.0, strokes[1][0].Y);
            Assert.AreEqual(251.0, strokes[1][1].X);
            Assert.AreEqual(227.0, strokes[1][1].Y);
            Assert.AreEqual(251.0, strokes[1][2].X);
            Assert.AreEqual(227.0, strokes[1][2].Y);
            Assert.AreEqual(253.0, strokes[1][3].X);
            Assert.AreEqual(230.0, strokes[1][3].Y);
            Assert.AreEqual(255.0, strokes[1][4].X);
            Assert.AreEqual(239.0, strokes[1][4].Y);
            Assert.AreEqual(256.0, strokes[1][5].X);
            Assert.AreEqual(245.0, strokes[1][5].Y);
            Assert.AreEqual(258.0, strokes[1][6].X);
            Assert.AreEqual(255.0, strokes[1][6].Y);
        }

        /// <summary>
        /// Tests a signature with a card swipe happening in the middle of it.
        /// </summary>
        [TestMethod]
        public void TestCompressedHighStress() {

            Signature signature = eScriptHR.ParseSignature("~STSIGN D ST1501-STN 65.43.21 \"{o[+:mv+T]z!^s.M*svCHs-%_3vAN'h%)avQHqrABab+_a-EFadE_e6ABq.<Lo.Y_a6O{]z+TaxIRc-\"JsjWR1pA|yzCJ3xI|q[M_en}LsjWV `");

            Assert.AreEqual("ST1501-STN", signature.GetModel());
            Assert.AreEqual("D", signature.GetProtocolVersion());
            Assert.AreEqual("65.43.21", signature.GetVersion());

            Assert.AreEqual(4, signature.GetStrokes().Count);

            Assert.AreEqual(17, signature.GetStrokes()[0].Count);

            Assert.AreEqual(191.0,  Math.Round(signature.GetStrokes()[0][0].X));
            Assert.AreEqual(59.0,   Math.Round(signature.GetStrokes()[0][0].Y));
            Assert.AreEqual(191.0,  Math.Round(signature.GetStrokes()[0][1].X));
            Assert.AreEqual(59.0,   Math.Round(signature.GetStrokes()[0][1].Y));
            Assert.AreEqual(252.0,  Math.Round(signature.GetStrokes()[0][2].X));
            Assert.AreEqual(119.0,  Math.Round(signature.GetStrokes()[0][2].Y));
            Assert.AreEqual(312.0,  Math.Round(signature.GetStrokes()[0][3].X));
            Assert.AreEqual(119.0,  Math.Round(signature.GetStrokes()[0][3].Y));
            Assert.AreEqual(371.0,  Math.Round(signature.GetStrokes()[0][4].X));
            Assert.AreEqual(119.0,  Math.Round(signature.GetStrokes()[0][4].Y));
            Assert.AreEqual(433.0,  Math.Round(signature.GetStrokes()[0][5].X));
            Assert.AreEqual(120.0,  Math.Round(signature.GetStrokes()[0][5].Y));
            Assert.AreEqual(496.0,  Math.Round(signature.GetStrokes()[0][6].X));
            Assert.AreEqual(121.0,  Math.Round(signature.GetStrokes()[0][6].Y));
            Assert.AreEqual(1520.0, Math.Round(signature.GetStrokes()[0][7].X));
            Assert.AreEqual(120.0,  Math.Round(signature.GetStrokes()[0][7].Y));
            Assert.AreEqual(1520.0, Math.Round(signature.GetStrokes()[0][8].X));
            Assert.AreEqual(115.0,  Math.Round(signature.GetStrokes()[0][8].Y));
            Assert.AreEqual(1520.0, Math.Round(signature.GetStrokes()[0][9].X));
            Assert.AreEqual(120.0,  Math.Round(signature.GetStrokes()[0][9].Y));
            Assert.AreEqual(1520.0, Math.Round(signature.GetStrokes()[0][10].X));
            Assert.AreEqual(125.0,  Math.Round(signature.GetStrokes()[0][10].Y));
            Assert.AreEqual(1520.0, Math.Round(signature.GetStrokes()[0][11].X));
            Assert.AreEqual(130.0,  Math.Round(signature.GetStrokes()[0][11].Y));
            Assert.AreEqual(1520.0, Math.Round(signature.GetStrokes()[0][12].X));
            Assert.AreEqual(125.0,  Math.Round(signature.GetStrokes()[0][12].Y));
            Assert.AreEqual(1518.0, Math.Round(signature.GetStrokes()[0][13].X));
            Assert.AreEqual(120.0,  Math.Round(signature.GetStrokes()[0][13].Y));
            Assert.AreEqual(1516.0, Math.Round(signature.GetStrokes()[0][14].X));
            Assert.AreEqual(115.0,  Math.Round(signature.GetStrokes()[0][14].Y));
            Assert.AreEqual(1518.0, Math.Round(signature.GetStrokes()[0][15].X));
            Assert.AreEqual(110.0,  Math.Round(signature.GetStrokes()[0][15].Y));
            Assert.AreEqual(1516.0, Math.Round(signature.GetStrokes()[0][16].X));
            Assert.AreEqual(115.0,  Math.Round(signature.GetStrokes()[0][16].Y));
        
            Assert.AreEqual(2, signature.GetStrokes()[1].Count);

            Assert.AreEqual(0.0,    Math.Round(signature.GetStrokes()[1][0].X));
            Assert.AreEqual(0.0,    Math.Round(signature.GetStrokes()[1][0].Y));
            Assert.AreEqual(2999.0, Math.Round(signature.GetStrokes()[1][1].X));
            Assert.AreEqual(2999.0, Math.Round(signature.GetStrokes()[1][1].Y));
        
            Assert.AreEqual(2, signature.GetStrokes()[2].Count);

            Assert.AreEqual(0.0,    Math.Round(signature.GetStrokes()[2][0].X));
            Assert.AreEqual(2999.0, Math.Round(signature.GetStrokes()[2][0].Y));
            Assert.AreEqual(2999.0, Math.Round(signature.GetStrokes()[2][1].X));
            Assert.AreEqual(0.0,    Math.Round(signature.GetStrokes()[2][1].Y));
        
            Assert.AreEqual(5, signature.GetStrokes()[3].Count);
        
            Assert.AreEqual(0.0,    Math.Round(signature.GetStrokes()[3][0].X));
            Assert.AreEqual(0.0,    Math.Round(signature.GetStrokes()[3][0].Y));
            Assert.AreEqual(2999.0, Math.Round(signature.GetStrokes()[3][1].X));
            Assert.AreEqual(0.0,    Math.Round(signature.GetStrokes()[3][1].Y));
            Assert.AreEqual(2999.0, Math.Round(signature.GetStrokes()[3][2].X));
            Assert.AreEqual(2999.0, Math.Round(signature.GetStrokes()[3][2].Y));
            Assert.AreEqual(0.0,    Math.Round(signature.GetStrokes()[3][3].X));
            Assert.AreEqual(2999.0, Math.Round(signature.GetStrokes()[3][3].Y));
            Assert.AreEqual(0.0,    Math.Round(signature.GetStrokes()[3][4].X));
            Assert.AreEqual(0.0,    Math.Round(signature.GetStrokes()[3][4].Y));
        }

        /// <summary>
        /// This method tests that track1 and track3 of an ID card are parsed correctly
        /// </summary>
        [TestMethod]
        public void TestIdentificationCard1()
        {
            string swipe = "!STCARD A %OHWEST LIBERTY^SMITH$STEVE$W$^100 EAST AVE^?;1234567890123456=141156780912?%12345678901  D B         M   1600180BLNBLU                          #Z#-@)     ?\r";
            CardSwipe cs = eScriptLR.ParseCardSwipe(swipe);
            FinancialCard fc = cs.GetFinancialCard();
            IdentificationCard ic = cs.GetIdentificationCard();

            FinancialCardTrackOne fc1 = null;
            FinancialCardTrackTwo fc2 = null;
            if (fc != null)
            {
                fc1 = fc.GetTrackOne();
                fc2 = fc.GetTrackTwo();
            }

            IdentificationCardTrackOne ic1 = null;
            IdentificationCardTrackTwo ic2 = null;
            IdentificationCardTrackThree ic3 = null;

            if (ic != null)
            {
                ic1 = ic.GetTrackOne();
                ic2 = ic.GetTrackTwo();
                ic3 = ic.GetTrackThree();
            }

            Assert.IsTrue(ic1.GetFirstName().Equals("STEVE"));
            Assert.IsTrue(ic1.GetLastName().Equals("SMITH"));
            Assert.IsTrue(ic1.GetMiddleName().Equals("W"));
            Assert.IsTrue(ic1.GetAddress().Equals("100 EAST AVE"));
            Assert.IsTrue(ic1.GetCity().Equals("WEST LIBERTY"));
            Assert.IsTrue(ic1.GetState().Equals("OH"));
            Assert.IsTrue(ic3.GetEyeColor().Equals("BLU"));
            Assert.IsTrue(ic3.GetHairColor().Equals("BLN"));
            Assert.IsTrue(ic3.GetHeight().Equals("600"));
            Assert.IsTrue(ic3.GetWeight().Equals("180"));
            Assert.IsTrue(ic3.GetLicenseClass().Equals("D"));
            Assert.IsTrue(ic3.GetSex() == IdentificationSex.MALE);
            Assert.IsTrue(ic2 == null);
        }

        /// <summary>
        /// This method tests that track1 of an ID card is parsed correctly
        /// </summary>
        [TestMethod]
        public void TestIdentificationCard2()
        {
            string swipe = "!STCARD A %CALOS ANGELES^DOE$JOHN$CARL^14324 MAIN ST APT 5^?\r";
            CardSwipe cs = eScriptLR.ParseCardSwipe(swipe);
            FinancialCard fc = cs.GetFinancialCard();
            IdentificationCard ic = cs.GetIdentificationCard();

            FinancialCardTrackOne fc1 = null;
            FinancialCardTrackTwo fc2 = null;
            if (fc != null)
            {
                fc1 = fc.GetTrackOne();
                fc2 = fc.GetTrackTwo();
            }

            IdentificationCardTrackOne ic1 = null;
            IdentificationCardTrackTwo ic2 = null;
            IdentificationCardTrackThree ic3 = null;

            if (ic != null)
            {
                ic1 = ic.GetTrackOne();
                ic2 = ic.GetTrackTwo();
                ic3 = ic.GetTrackThree();
            }

            Assert.IsTrue(fc == null);
            Assert.IsTrue(ic1.GetFirstName().Equals("JOHN"));
            Assert.IsTrue(ic1.GetLastName().Equals("DOE"));
            Assert.IsTrue(ic1.GetMiddleName().Equals("CARL"));
            Assert.IsTrue(ic1.GetAddress().Equals("14324 MAIN ST APT 5"));
            Assert.IsTrue(ic2 == null);
            Assert.IsTrue(ic3 == null);
            Assert.IsTrue(ic1.GetCity().Equals("LOS ANGELES"));
            Assert.IsTrue(ic1.GetState().Equals("CA"));
        }

        /// <summary>
        /// This method tests that all tracks of an ID card are parsed correctly
        /// </summary>
        [TestMethod]
        public void TestIdentificationCard3()
        {
            string swipe = "!STCARD A %OHWEST LIBERTY^SMITH$STEVE$W$^100 EAST AVE^?;655555111222333=160319850603?;1234567890123456=141156780912?%12345678901  D B         M   1600180BLNBLU                          #Z#-@)     ?\r";
            CardSwipe cs = eScriptLR.ParseCardSwipe(swipe);
            FinancialCard fc = cs.GetFinancialCard();
            IdentificationCard ic = cs.GetIdentificationCard();

            FinancialCardTrackOne fc1 = null;
            FinancialCardTrackTwo fc2 = null;
            if (fc != null)
            {
                fc1 = fc.GetTrackOne();
                fc2 = fc.GetTrackTwo();
            }

            IdentificationCardTrackOne ic1 = null;
            IdentificationCardTrackTwo ic2 = null;
            IdentificationCardTrackThree ic3 = null;

            if (ic != null)
            {
                ic1 = ic.GetTrackOne();
                ic2 = ic.GetTrackTwo();
                ic3 = ic.GetTrackThree();
            }

            Assert.IsTrue(ic1.GetFirstName().Equals("STEVE"));
            Assert.IsTrue(ic1.GetLastName().Equals("SMITH"));
            Assert.IsTrue(ic1.GetMiddleName().Equals("W"));
            Assert.IsTrue(ic1.GetAddress().Equals("100 EAST AVE"));
            Assert.IsTrue(ic1.GetCity().Equals("WEST LIBERTY"));
            Assert.IsTrue(ic1.GetState().Equals("OH"));

            Assert.IsTrue(ic3.GetEyeColor().Equals("BLU"));
            Assert.IsTrue(ic3.GetHairColor().Equals("BLN"));
            Assert.IsTrue(ic3.GetHeight().Equals("600"));
            Assert.IsTrue(ic3.GetWeight().Equals("180"));
            Assert.IsTrue(ic3.GetLicenseClass().Equals("D"));
            Assert.IsTrue(ic3.GetSex() == IdentificationSex.MALE);

            Assert.IsTrue(ic2.GetBirthDate().Month == 6);
            Assert.IsTrue(ic2.GetBirthDate().Year == 1985);
            Assert.IsTrue(ic2.GetBirthDate().Day == 3);
            Assert.IsTrue(ic2.GetExpiration().Month == 3);
            Assert.IsTrue(ic2.GetExpiration().Year == 2016);
            Assert.IsTrue(ic2.GetExpiration().Day == 31);
            Assert.IsTrue(ic2.GetExpiration().Hour == 23);
            Assert.IsTrue(ic2.GetExpiration().Minute == 59);
            Assert.IsTrue(ic2.GetExpiration().Second == 59);
            Assert.IsTrue(ic2.GetIdNumber().Equals("111222333"));
            Assert.IsTrue(ic2.GetIssuerNumber().Equals("655555"));
        }

        /// <summary>
        /// Tests to make sure the library can properly handle financial cards.
        /// </summary>
        [TestMethod]
        public void TestFinancialCard()
        {
            string swipe = "!STCARD A %B6131380999977777^SMITH/STEVE J^170956789012?;1234567890123456=17095678901234567890?\r";
        
            CardSwipe cs = null;
            try
            {
                cs = eScriptLR.ParseCardSwipe(swipe);
            }
            catch (CardSwipeInvalidException)
            {
                Assert.IsTrue(false);
            }
        
            IdentificationCard ic = cs.GetIdentificationCard();
            FinancialCard fc = cs.GetFinancialCard();

            Assert.IsTrue(ic == null);

            Assert.IsTrue(fc != null);
            Assert.IsTrue(fc.GetTrackOne().GetAccountNumber().Equals("6131380999977777"));
            Assert.IsTrue(fc.GetTrackOne().GetFirstName().Equals("STEVE J"));
            Assert.IsTrue(fc.GetTrackOne().GetLastName().Equals("SMITH"));
            DateTime expDate = fc.GetTrackOne().GetExpiration();

            Assert.IsTrue(expDate.Year == 2017);
            Assert.IsTrue(expDate.Month == 9);

            Assert.IsTrue(fc.GetTrackTwo().GetAccountNumber().Equals("1234567890123456"));
        
            expDate = fc.GetTrackTwo().GetExpiration();

            Assert.IsTrue(expDate.Year == 2017);
            Assert.IsTrue(expDate.Month == 9);
        }

        /// <summary>
        /// Tests to make sure the expiration parser properly handles wrap around.
        /// </summary>
        [TestMethod]
        public void TestExpirationWrapAround()
        {
            string swipe = "!STCARD A %B6131380999977777^SMITH/STEVE J^171256789012?;1234567890123456=17125678901234567890?\r";

            CardSwipe cs = null;
            try
            {
                cs = eScriptLR.ParseCardSwipe(swipe);
            }
            catch (CardSwipeInvalidException)
            {
                Assert.IsTrue(false);
            }

            IdentificationCard ic = cs.GetIdentificationCard();
            FinancialCard fc = cs.GetFinancialCard();

            Assert.IsTrue(ic == null);

            Assert.IsTrue(fc != null);
            Assert.IsTrue(fc.GetTrackOne().GetAccountNumber().Equals("6131380999977777"));
            Assert.IsTrue(fc.GetTrackOne().GetFirstName().Equals("STEVE J"));
            Assert.IsTrue(fc.GetTrackOne().GetLastName().Equals("SMITH"));
            DateTime expDate = fc.GetTrackOne().GetExpiration();

            Assert.IsTrue(expDate.Year == 2017);
            Assert.IsTrue(expDate.Month == 12);

            Assert.IsTrue(fc.GetTrackTwo().GetAccountNumber().Equals("1234567890123456"));

            expDate = fc.GetTrackTwo().GetExpiration();

            Assert.IsTrue(expDate.Year == 2017);
            Assert.IsTrue(expDate.Month == 12);
        }
    }
}
