<?php
/**
 * This file contains the {@link IdentificationCardTrackThree} class.
 * @copyright (c) 2015, Scriptel Corporation.
 */
namespace com\scriptel;

/**
 * This class represents track three of a magnetic strip identification card
 * such as a United States driver's license.
 */
class IdentificationCardTrackThree {
    /**
     * CDS protocol version number.
     * @var string
     */
    private $cdsVersion;
    /**
     * Jurisdiction version number.
     * @var string
     */
    private $jurisdictionVersion;
    /**
     * Postal code of the card holder.
     * @var string
     */
    private $postalCode;
    /**
     * License class
     * @var string
     */
    private $licenseClass;
    /**
     * License restrictions.
     * @var string
     */
    private $restrictions;
    /**
     * License endorsements.
     * @var string
     */
    private $endorsements;
    /**
     * Card-holder gender.
     * @var IdentificationSex
     */
    private $sex;
    /**
     * Card-holder height.
     * @var string
     */
    private $height;
    /**
     * Card-holder weight.
     * @var string
     */
    private $weight;
    /**
     * Card-holder hair color.
     * @var string
     */
    private $hairColor;
    /**
     * Card-holder eye color.
     * @var string
     */
    private $eyeColor;
    /**
     * Vendor-Specific discretionary data.
     * @var string
     */
    private $discretionaryData;


    /**
     * This method attempts to parse track three of an identification card.
     * @param string $trackData
     * @return \IdentificationTrackThree
     */
    public static function parse($trackData) {
        $myPattern = '/%(.{1})([0-9 ]{1})([A-Z 0-9]{11})([A-Z 0-9]{2})([A-Z 0-9]{10})([A-Z 0-9]{4})(1|2|M|F)([0-9 ]{3})([0-9 ]{3})([A-Z ]{3})([A-Z ]{3})([^\?]{0,37})\?/';
        $matches= array();
        preg_match($myPattern, $trackData, $matches);
        if (count($matches) > 0 ) {
            $track = new IdentificationCardTrackThree();
            $track->setCdsVersion($matches[1]);
            $track->setJurisdictionVersion($matches[2])*1;
            $track->setPostalCode(trim($matches[3]));
            $track->setLicenseClass(trim($matches[4]));
            $track->setRestrictions(trim($matches[5]));
            $track->setEndorsements(trim($matches[6]));
            $track->setSex(($matches[7]=="1"||$matches[7]=="M")?"M":"F");
            $track->setHeight(trim($matches[8]));
            $track->setWeight(trim($matches[9]));
            $track->setHairColor(trim($matches[10]));
            $track->setEyeColor(trim($matches[11]));
            $track->setDiscretionaryData($matches[12]);

            return $track;
        }
        return NULL;
    }
    /**
     * Returns this object as a human readable string.
     * @return string
     */
    public function toString() {
        $strCardData = "";
        $strCardData .= "Identification Card - Track Three: \n";
        $strCardData .= "         CDS Version: " . $this->getCdsVersion() . "\n";
        $strCardData .= "Jurisdiction Version: " . $this->getJurisdictionVersion() . "\n";
        $strCardData .= "         Postal Code: " . $this->getPostalCode() . "\n";
        $strCardData .= "       License Class: " . $this->getLicenseClass() . "\n";
        $strCardData .= "        Restrictions: " . $this->getRestrictions() . "\n";
        $strCardData .= "        Endorsements: " . $this->getEndorsements() . "\n";
        $strCardData .= "                 Sex: " . $this->getSex() . "\n";
        $strCardData .= "              Height: " . $this->getHeight() . "\n";
        $strCardData .= "              Weight: " . $this->getWeight() . "\n";
        $strCardData .= "          Hair Color: " . $this->getHairColor() . "\n";
        $strCardData .= "           Eye Color: " . $this->getEyeColor() . "\n";
        $strCardData .= "  Discretionary Data: " . $this->getDiscretionaryData() . "\n";
        return $strCardData;
     }

    /**
     * Gets the CDS protocol version number.
     * @return string
     */
    function getCdsVersion() {
        return $this->cdsVersion;
    }
    /**
     * Gets the jurisdiction version from the card.
     * @return string
     */
    function getJurisdictionVersion() {
        return $this->jurisdictionVersion;
    }
    /**
     * Gets the postal code from the card.
     * @return string
     */
    function getPostalCode() {
        return $this->postalCode;
    }
    /**
     * Gets the license class.
     * @return string
     */
    function getLicenseClass() {
        return $this->licenseClass;
    }
    /**
     * Gets the license restrictions.
     * @return string
     */
    function getRestrictions() {
        return $this->restrictions;
    }
    /**
     * Gets license endorsements.
     * @return string
     */
    function getEndorsements() {
        return $this->endorsements;
    }
    /**
     * Gets card-holder gender.
     * @return IdentificationSex
     */
    function getSex() {
        return $this->sex;
    }
    /**
     * Gets the card-holder height.
     * @return string
     */
    function getHeight() {
        return $this->height;
    }
    /**
     * Gets the card-holder weight.
     * @return string
     */
    function getWeight() {
        return $this->weight;
    }
    /**
     * Gets the card-holder hair color.
     * @return string
     */
    function getHairColor() {
        return $this->hairColor;
    }
    /**
     * Gets the card-holder eye color.
     * @return string
     */
    function getEyeColor() {
        return $this->eyeColor;
    }
    /**
     * Gets vendor-specific discretionary data.
     * @return string
     */
    function getDiscretionaryData() {
        return $this->discretionaryData;
    }
    /**
     * Sets the CDS protocol version number.
     * @param string $cdsVersion
     */
    function setCdsVersion($cdsVersion) {
        $this->cdsVersion = $cdsVersion;
    }
    /**
     * Sets the jurisdiction version on the card.
     * @param string $jurisdictionVersion
     */
    function setJurisdictionVersion($jurisdictionVersion) {
        $this->jurisdictionVersion = $jurisdictionVersion;
    }
    /**
     * Sets the postal code from the card.
     * @param string $postalCode
     */
    function setPostalCode($postalCode) {
        $this->postalCode = $postalCode;
    }
    /**
     * Sets the license class.
     * @param string $licenseClass
     */
    function setLicenseClass($licenseClass) {
        $this->licenseClass = $licenseClass;
    }
    /**
     * Sets the license restrictions.
     * @param string $restrictions
     */
    function setRestrictions($restrictions) {
        $this->restrictions = $restrictions;
    }
    /**
     * Sets license endorsements.
     * @param string $endorsements
     */
    function setEndorsements($endorsements) {
        $this->endorsements = $endorsements;
    }
    /**
     * Sets the card-holder gender.
     * @param IdentificationSex $sex
     */
    function setSex($sex) {
        $this->sex = $sex;
    }
    /**
     * Sets the card-holder height.
     * @param string $height
     */
    function setHeight($height) {
        $this->height = $height;
    }
    /**
     * Sets the card-holder weight.
     * @param string $weight
     */
    function setWeight($weight) {
        $this->weight = $weight;
    }
    /**
     * Sets the card-holder hair color.
     * @param string $hairColor
     */
    function setHairColor($hairColor) {
        $this->hairColor = $hairColor;
    }
    /**
     * Sets the card-holder eye color.
     * @param string $eyeColor
     */
    function setEyeColor($eyeColor) {
        $this->eyeColor = $eyeColor;
    }
    /**
     * Sets vendor-specific discretionary data.
     * @param string $discretionaryData
     */
    function setDiscretionaryData($discretionaryData) {
        $this->discretionaryData = $discretionaryData;
    }
}
