<?php
/**
 * This file contains the {@link IdentificationCard} class.
 * @copyright (c) 2015, Scriptel Corporation.
 */
namespace com\scriptel;

/**
 * This class represents an identification card such as a United States drivers
 * license.
 * @see <a href="http://www.aamva.org/?aspxerrorpath=/NR/rdonlyres/66260AD6-64B9-45E9-A253-B8AA32241BE0/0/2005DLIDCardSpecV2FINAL.pdf">http://www.aamva.org/?aspxerrorpath=/NR/rdonlyres/66260AD6-64B9-45E9-A253-B8AA32241BE0/0/2005DLIDCardSpecV2FINAL.pdf</a>
 */
class IdentificationCard {
    /**
     * Identification Card Track One.
     * @var IdentificationCardTrackOne.
     */
    private $trackOne;
    /**
     * Identification Card Track Two.
     * @var IdentificationCardTrackTwo.
     */
    private $trackTwo;
    /**
     * Identification Card Track Three.
     * @var IdentificationCardTrackThree.
     */
    private $trackThree;

    /**
     * This method attempts to parse identification card information out of a
     * raw magnetic strip string.
     * @param $cardData Card data to parse.
     * @return Identification Card data if parsed, null otherwise.
     */
    public static function parse($cardData) {
        $track1 = IdentificationCardTrackOne::parse($cardData);
        $track2 = IdentificationCardTrackTwo::parse($cardData);
        $track3 = IdentificationCardTrackThree::parse($cardData);

        if ($track1 != NULL || $track2 != NULL || $track3 != NULL) {
            $card = new IdentificationCard();
            $card->setTrackOne($track1);
            $card->setTrackTwo($track2);
            $card->setTrackThree($track3);
            return $card;
        }
        return NULL;
    }

    /**
     * Returns this object as a human readable string.
     * @return string
     */
    public function toString() {
        $strCardData = "";
        $strCardData .= "Identification Card \n";
        if ($this->trackOne != NULL) {
            $strCardData .= $this->getTrackOne()->toString() . "\n";
        }
        if ($this->trackTwo != NULL) {
            $strCardData .= $this->getTrackTwo()->toString() . "\n";
        }
        if ($this->trackThree != NULL) {
            $strCardData .= $this->getTrackThree()->toString() . "\n";
        }
        return $strCardData;
    }
    /**
     * Gets parsed track one data.
     * @return IdentificationCardTrackOne Track one data.
     */
    function getTrackOne() {
        return $this->trackOne;
    }
    /**
     * Gets parsed track two data.
     * @return IdentificationCardTrackTwo Track two data.
     */
    function getTrackTwo() {
        return $this->trackTwo;
    }
    /**
     * Gets parsed track three data.
     * @return IdentificationCardTrackThree Track three data.
     */
    function getTrackThree() {
        return $this->trackThree;
    }
    /**
     * Sets parsed track one data.
     * @param IdentificationCardTrackOne $trackOne
     */
    function setTrackOne($trackOne) {
        $this->trackOne = $trackOne;
    }
    /**
     * Sets parsed track two data.
     * @param IdentificationCardTrackTwo $trackTwo
     */
    function setTrackTwo($trackTwo) {
        $this->trackTwo = $trackTwo;
    }
    /**
     * Sets parsed track three data.
     * @param IdentificationCardTrackThree $trackThree
     */
    function setTrackThree($trackThree) {
        $this->trackThree = $trackThree;
    }
}
