<?php
/**
 * This file contains the {@link FinancialCardTrackTwo} class.
 * @copyright (c) 2015, Scriptel Corporation.
 */
namespace com\scriptel;

/**
 * This class represents the second track of a financial card.
 */
class FinancialCardTrackTwo {
    /**
     * Credit card number.
     * @var string
     */
    private $accountNumber;
    /**
     * Expiration date.
     * @var Date
     */
    private $expiration;
    /**
     * Service code.
     * @var string
     */
    private $serviceCode;
    /**
     * Discretionary Data
     * @var string
     */
    private $discretionaryData;

    /**
     * This method attempts to parse track two of the raw track string.
     * @param string $trackData Track data to parse.
     * @return FinancialCardTrackTwo data if data found, null otherwise.
     */
    public static function parse($trackData) {
        $myPattern = ".;(\\d{1,19})=(\\d{2})(\\d{2})(\\d{3})([A-Za-z 0-9]*)\\?.";
        $matches[] = NULL;
        preg_match($myPattern, $trackData, $matches);
        if (count($matches) > 0 ) {
            $track = new FinancialCardTrackTwo();
            $track->setAccountNumber($matches[1]);
            $track->setServiceCode($matches[4]);
            $track->setDiscretionaryData($matches[5]);
            // Create the expiration date and time.
            $expMonth = ($matches[3]*1);
            $expYear = ($matches[2]*1)+2000;
            $track->setExpiration(mktime(0,0,0,$expMonth+1,1,$expYear)-1);
            return $track;
        }
        return NULL;
    }
    /**
     * Returns this object as a human readable string.
     */
    public function toString() {
        $strCardData = "";
        $strCardData .= "Financial Card - Track Two: \n";
        $strCardData .= "    Account Number: " . $this->getAccountNumber() . "\n";
        $strCardData .= "        Expiration: " . $this->getExpiration() . "\n";
        $strCardData .= "      Service Code: " . $this->getServiceCode() . "\n";
        $strCardData .= "Discretionary Data: " . $this->getDiscretionaryData() . "\n";
        return $strCardData;
    }

    /**
     * Gets the account number of the card in track two.
     * @return string
     */
    function getAccountNumber() {
        return $this->accountNumber;
    }
    /**
     * Gets the expiration date of the card.
     * @return Date
     */
    function getExpiration() {
        return $this->expiration;
    }
    /**
     * Gets the service code on the card.
     * @return string
     */
    function getServiceCode() {
        return $this->serviceCode;
    }
    /**
     * Gets the discretionary data on the second track of the card.
     * @return string
     */
    function getDiscretionaryData() {
        return $this->discretionaryData;
    }
    /**
     * Sets the account number of the card in track two.
     * @param string $accountNumber
     */
    function setAccountNumber($accountNumber) {
        $this->accountNumber = $accountNumber;
    }
    /**
     * Sets the expiration date of the card.
     * @param Date $expiration
     */
    function setExpiration($expiration) {
        $this->expiration = $expiration;
    }
    /**
     * Sets the service code on the card.
     * @param string $serviceCode
     */
    function setServiceCode($serviceCode) {
        $this->serviceCode = $serviceCode;
    }
    /**
     * Sets the discretionary data on the second track of the card.
     * @param string $discretionaryData
     */
    function setDiscretionaryData($discretionaryData) {
        $this->discretionaryData = $discretionaryData;
    }
}
