<?php
/**
 * This file contains the {@link FinancialCardIssuer} class.
 * @copyright (c) 2015, Scriptel Corporation.
 */
namespace com\scriptel;

/**
 * This class attempts to identify credit card issuers based on patterns in the
 * numbers each vendor issues.
 */
final class FinancialCardIssuer {
    const AMERICAN_EXPRESS = "/^(34|37)\\d{13}$/";
    const DINERS_CLUB = "/^(30[0-5]\\d{11}|36\\d{12})$/";
    const CARTE_BLANCH = "/^38\\d{12}$/";
    const DISCOVER = "/^6011\\d{12}$/";
    const ENROUTE = "/^(2014|2149)\\d{11}$/";
    const JCB = "/^(3\\d{15}|(2131|1800)\\d{11})$/";
    const MASTER_CARD = "/^5[1-5]\\d{14}$/";
    const VISA = "/^4\\d{12,15}$/";
    const UNKNOWN = "/./";

    /**
     * Contains the set issuer name.
     */
    private $issuerName;

    /**
     * Constructor, creates a new instance of this class.
     * @param string issuerName The issuer name.
     */
    public function __construct($issuerName) {
        $this->issuerName = $issuerName;
    }

    /**
     * This method attempts to identify the issuer of a credit card based on
     * a credit card number.
     * @param string $cardNumber Number to inspect.
     * @return FinancialCardIssuer Issuer if identified, UNKNOWN otherwise.
     */
    public static function identifyByCardNumber($cardNumber) {
        $fci = new FinancialCardIssuer(FinancialCardIssuer::UNKNOWN);
        $myClass = new \ReflectionClass($fci);
        $list = $myClass->getConstants();
        $matches = NULL;
        foreach ($list as $issuerName=>$regex) {
            preg_match($regex, $cardNumber, $matches);
            if (count($matches) > 0) {
                $fci->setIssuerName($issuerName);
                return $fci;
            }
        }
    }

    /**
     * Gets the issuer name.
     * @return string Human-friendly issuer name.
     */
    function getIssuerName() {
        return $this->issuerName;
    }
    /**
     * Sets the issuer name.
     * @param string $issuerName
     */
    function setIssuerName($issuerName) {
        $this->issuerName = $issuerName;
    }
}






