<?php
/**
 * This file contains the {@link BatchListener} class.
 * @copyright (c) 2015, Scriptel Corporation.
 */
namespace com\scriptel;

/**
 * This class is used to collect and store events so that we can give an
 * entire signature at once without callers needing to implement an event
 * listener.
 */
class BatchListener implements EasyScriptEventListener {
    /**
     * Contains a list of strokes
     * @var $strokes stroke[]
     */
    private $strokes = array();
    /**
     * Contains a list of coordinates.
     * @var $stroke Coordinate[]
     */
    private $stroke;
    /**
     * Whether or not this signature is complete (canceled or otherwise).
     * @var boolean
     */
    private $done = FALSE;
    /**
     * Whether or not this signature was canceled.
     * @var boolean
     */
    private $canceled = FALSE;
    /**
     * A card swipe, if one is detected.
     * @var CardSwipe
     */
    private $swipe;
    /**
     * Metadata from the signature.
     * @var SignatureMetaData
     */
    private $metaData;

    /**
     * This method gets the list of strokes
     * @return Coordinate[][].
     */
    public function getStrokes(){
        return $this->strokes;
    }
    /**
     * Gets the signature metadata.
     * @param SignatureMetaData $metaData
     */
    public function signatureMetaData($metaData) {
        $this->metaData = $metaData;
    }
    /**
     * Marks a signature canceled and done.
     */
    public function cancel() {
        $this->done = TRUE;
        $this->canceled = TRUE;
        $this->strokes = array();
    }
    /**
     * Marks a signature complete.
     */
    public function endOfSignature() {
        $this->done = TRUE;
    }
    /**
     * Adds a new stroke to the stack.
     */
    public function newStroke() {
        $this->stroke = array();
        array_push($this->strokes, $this->stroke);

    }
    /**
     * Gets a coordinate that will be added to the current stroke.
     * @param Coordinate $coordinate
     */
    public function receiveCoordinate($coordinate) {
        $this->strokes[count($this->strokes)-1][] = $coordinate;
    }
    /**
     * Captures a card swipe.
     * @param CardSwipe $swipe
     */
    public function cardSwipe($swipe) {
        $this->swipe = $swipe;
    }
    /**
     * Gets whether or not this signature is complete (canceled or otherwise).
     * @return boolean
     */
    function isDone(){
        return $this->done;
    }
    /**
     * Gets whether or not this signature was canceled.
     * @return boolean
     */
    function isCanceled(){
        return $this->canceled;
    }

    /**
     * Gets a card swipe if one was found.
     * @return CardSwipe.
     */
    function getSwipe() {
        return $this->swipe;
    }

    /**
     * Gets the meta data of the signature.
     * @return SignatureMetaData.
     */
    function getMetaData(){
        return $this->metaData;
    }
}