/*
 * Copyright 2013 - Scriptel Corporation
 */
package com.scriptel.easyscript;

import java.util.Calendar;
import java.util.Date;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * This class represents the second track of a financial card.
 */
public class FinancialCardTrackTwo {
    /**
     * Credit card number.
     */
    private String accountNumber;
    /**
     * Card expiration date.
     */
    private Date expiration;
    /**
     * Service code.
     */
    private String serviceCode;
    /**
     * Discretionary data.
     */
    private String discretionaryData;

    /**
     * This method attempts to parse track two out the raw track string.
     * @param trackData Track data to parse.
     * @return FinancialCardTrackTwo data if data found, null otherwise.
     */
    public static FinancialCardTrackTwo parse(String trackData) {
        Pattern p = Pattern.compile(".*;(\\d{1,19})=(\\d{2})(\\d{2})(\\d{3})([A-Za-z 0-9]*)\\?.*");
        Matcher m = p.matcher(trackData);
        if(m.matches()) {
            FinancialCardTrackTwo track = new FinancialCardTrackTwo();
            track.setAccountNumber(m.group(1));
            track.setServiceCode(m.group(4));
            track.setDiscretionaryData(m.group(5));

            Calendar c = Calendar.getInstance();
            c.set(Calendar.YEAR, Integer.parseInt(m.group(2))+2000);
            c.set(Calendar.MONTH, Integer.parseInt(m.group(3)));
            c.set(Calendar.DAY_OF_MONTH, 1);
            c.set(Calendar.HOUR_OF_DAY, 0);
            c.set(Calendar.MINUTE, 0);
            c.set(Calendar.SECOND, 0);
            c.set(Calendar.MILLISECOND, 0);
            
            c.add(Calendar.MILLISECOND, -1);
            track.setExpiration(c.getTime());
            
            return track;
        }
        return null;
    }
    /**
     * Returns this object as a human readable string.
     * @return String containing human readable string.
     */
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        String ln = System.getProperty("line.separator");
        sb.append("Financial Card - Track Two:").append(ln);
        sb.append("    Account Number: ").append(accountNumber).append(ln);
        sb.append("        Expiration: ").append(expiration).append(ln);
        sb.append("      Service Code: ").append(serviceCode).append(ln);
        sb.append("Discretionary Data: ").append(discretionaryData).append(ln);
        return sb.toString();
    }
    /**
     * Gets the account number of the card in track two.
     * @return Account number
     */
    public String getAccountNumber() {
        return accountNumber;
    }

    /**
     * Sets the account number of the card in track two.
     * @param accountNumber the account number to set
     */
    private void setAccountNumber(String accountNumber) {
        this.accountNumber = accountNumber;
    }

    /**
     * Gets the expiration date of the card.
     * @return the expiration
     */
    public Date getExpiration() {
        return expiration;
    }

    /**
     * Sets the expiration date of the card.
     * @param expiration the expiration to set
     */
    private void setExpiration(Date expiration) {
        this.expiration = expiration;
    }

    /**
     * Gets the service code on the card.
     * @return the serviceCode
     */
    public String getServiceCode() {
        return serviceCode;
    }

    /**
     * Sets the service code on the card.
     * @param serviceCode the serviceCode to set
     */
    private void setServiceCode(String serviceCode) {
        this.serviceCode = serviceCode;
    }

    /**
     * Gets the discretionary data on the second track of the card.
     * @return the discretionary data
     */
    public String getDiscretionaryData() {
        return discretionaryData;
    }

    /**
     * Sets the discretionary data on the second track of the card.
     * @param discretionaryData the discretionaryData to set
     */
    private void setDiscretionaryData(String discretionaryData) {
        this.discretionaryData = discretionaryData;
    }
}
