/*
 * Copyright 2015 - Scriptel Corporation
 */
package com.scriptel.easyscript;

/**
 * This class implements a binary tree reader for the compression table used
 * to compress EasyScript signatures.
 */
public class BinaryTreeReader {
    /**
     * Our current location in the binary tree.
     */
    private int location;
    /**
     * The head node in the binary tree.
     */
    private static BinaryTree head = null;
    /**
     * The compression table used to decompress signatures.
     */
    private static final String COMPRESSION_TABLE = 
            "0011000000100000011101101000000001011100000000001010000010000000001000000000" +
            "1000100000001100010000000100010010000000111010010000001010110010001110000000" +
            "1011101000000100101000000000010010010000001001110000000010010000000001010100" +
            "0000001000101000000100000001111010000000000110001000000010100100000011110000" +
            "1000000101010100000001011110000000100110100000000011101000000001110100000000" +
            "1100000000100000011001110000001101110100000010001101000000111010100000011001" +
            "0001000000011011100000001110010000000010100100000011110101000011000000010000" +
            "0010011001000000100111001000110000000000100111100000001101001000000101111000" +
            "0000101000100000001010000000000100101100000010000001011010010000000000000010" +
            "0000010010101000000101110100000011000110000000100000010000000011011000000010" +
            "1100100000000010110000000100100000100000000011001000000010101100000010100001" +
            "0000000010001000000001111000001000000110110000010011000000001001101000000001" +
            "0101010000001011001000000011001100000001100010000000001010010000000101101000" +
            "0000010101110000001011011000000000010111110000001101000000000011000000000001" +
            "1011000000000011000010000001101101000000011001010000001010110000000000110001" +
            "1000000101100000000001101010000000110011100000000110010000000010100110000000" +
            "1101011000000101110000000010001000000000100000010001011011100000001000000000" +
            "0011000000111110000100000000100101000000011111010000001010011000000101111100" +
            "0010000000000010000001110010100000011010010000001110001000000101100010000001" +
            "1111110000001000010001000000110101100000010000010000000110100010000001100001" +
            "0000001001001000000011001";
    
    /**
     * This method sets up the compression tree.
     */
    public BinaryTreeReader() {
        if (head == null) {
            int bitDepth = readTableValue(6);
            head = writeTree(bitDepth);
        }
    }
    
    /**
     * This method finds that decimal value for some number of bits in the table starting at location.
     * @param ditDepth Number of bits to read.
     * @return The decimal value of the bits read.
     */
    private int readTableValue(int bitDepth) {
        char c;
        int v = 0;
        for (int i = 0; i < bitDepth; i++)  {
            c = COMPRESSION_TABLE.charAt(location++);
            v = (v * 2) + (c == '0'? 0 : 1);
        }
        return v;    
    }
    
    /**
     * This method sets values to the leafs in the compression binary tree.
     * @return The compressed binary tree.
     */
    private BinaryTree writeTree(int bitDepth) {
        BinaryTree node = new BinaryTree();
        int type = readTableValue(1);
        if (type == 0) {  // node found
            node.setLeft(writeTree(bitDepth));
            node.setRight(writeTree(bitDepth));
        } else {    // leaf found
            node.setValue(readTableValue(bitDepth));
        }
        return node;
    }

    /**
     * This method gets the head of the binary tree.
     * @return Gets the head of the binary tree.
     */
    public BinaryTree getTree() {
        return head;
    }
}
