﻿// <copyright file="FormExample.cs" company="Scriptel Corporation">
//      Copyright 2015 - Scriptel Corporation
// </copyright>
namespace FormExample
{
    using System;
    using System.Collections.Generic;
    using System.Data;
    using System.Drawing;
    using System.Drawing.Drawing2D;
    using System.Text;
    using System.Threading;
    using System.Windows.Forms;
    using ProScript;
    using System.Drawing.Imaging;

    public partial class FormExample : Form, IDeviceNotificationListener, IInputListener
    {
        /// <summary>
        /// Keeps track of the currently selected dvices.
        /// </summary>
        private Device currentDevice;
        /// <summary>
        /// Whether or not we believe we're currently connected.
        /// </summary>
        private bool connected = false;
        /// <summary>
        /// The thread we're going to use to read from the device.
        /// </summary>
        private Thread readThread;
        /// <summary>
        /// The graphics context of the Signature picture box.
        /// </summary>
        private Graphics graphics;
        /// <summary>
        /// The last coordinate we received from the device.
        /// </summary>
        private Coordinate lastCoordinate;
        /// <summary>
        /// The pen we'll use to draw on the graphics context.
        /// </summary>
        private Pen pen;
        /// <summary>
        /// The coordinate range of the currently connected device.
        /// </summary>
        private CoordinateRange range;
        /// <summary>
        /// Flag that tells us to clear the screen on the next coordinate.
        /// </summary>
        private bool clearOnNext = false;
        /// <summary>
        /// This object contains a bitmap with a copy of the signature pane.
        /// </summary>
        private Bitmap bitmap;

        /// <summary>
        /// Constructor, creates a new instance of this class.
        /// </summary>
        public FormExample()
        {
            InitializeComponent();
        }

        /// <summary>
        /// Form loader, initializes form components and enumerates devices.
        /// </summary>
        /// <param name="sender">Sender of the event.</param>
        /// <param name="e">Event arguments.</param>
        private void FormExample_Load(object sender, EventArgs e)
        {
            //Register this class as a device listener so we get notified
            //when devices are added to and removed from the system.
            DeviceManager.RegisterDeviceListener(this);

            //Set up our graphics context.
            bitmap = new Bitmap(signature.Width, signature.Height);
            graphics = Graphics.FromImage(bitmap);
            graphics.SmoothingMode = SmoothingMode.AntiAlias;
            pen = new Pen(Color.Black, 2);

            //Reload our device list.
            ReloadDeviceList();

            //If we've got a device, lets open it automatically.
            if (this.deviceSelect.Items.Count > 0)
            {
                OpenDevice();
            }
        }

        /// <summary>
        /// This method gets called when the connect/disconnect button is hit.
        /// </summary>
        /// <param name="sender">Event sender.</param>
        /// <param name="e">Event arguments.</param>
        private void connectButton_Click(object sender, EventArgs e)
        {
            if (connected)
            {
                CloseDevice();
            }
            else
            {
                OpenDevice();
            }
        }

        /// <summary>
        /// This method gets called when the form is closing.
        /// </summary>
        /// <param name="sender">Event sender.</param>
        /// <param name="e">Event arguments.</param>
        private void FormExample_FormClosing(object sender, FormClosingEventArgs e)
        {
            //Form is closing, make sure we close our device which will
            //stop the reading thread.
            CloseDevice();
        }

        /// <summary>
        /// Reloads the device list, repopulates by querying the ProScript library.
        /// Since this manipulates the UI it has to be called from the UI thread.
        /// </summary>
        private void ReloadDeviceList()
        {
            deviceSelect.Items.Clear();
            List<Device> devices = DeviceManager.GetAttachedDevices();
            foreach(Device d in devices) {
                DeviceComboWrapper wrapper = new DeviceComboWrapper(d);
                deviceSelect.Items.Add(wrapper);
                if (currentDevice != null && currentDevice.GetPath() == d.GetPath())
                {
                    //This one should be selected.
                    deviceSelect.SelectedItem = wrapper;
                }
            }

            if (deviceSelect.SelectedIndex < 0 && deviceSelect.Items.Count > 0)
            {
                deviceSelect.SelectedIndex = 0;
            }
        }

        /// <summary>
        /// This gets called when the selected device changes.
        /// </summary>
        /// <param name="sender">The sender of the event.</param>
        /// <param name="e">Event arguments.</param>
        private void deviceSelect_SelectedIndexChanged(object sender, EventArgs e)
        {
            //If any device is open close it.
            CloseDevice();
            
            DeviceComboWrapper wrapper = (DeviceComboWrapper)deviceSelect.SelectedItem;
            if (wrapper != null)
            {
                currentDevice = wrapper.Device;
            }
        }

        /// <summary>
        /// This method gets called when a new ScripTouch device is added to the system.
        /// </summary>
        /// <param name="path">Path of the added device.</param>
        /// <param name="device">Device that was added, may not be set.</param>
        public void ReceiveDeviceAttachedNotification(string path, Device device)
        {
            //Reload the device list on the main thread.
            this.Invoke((MethodInvoker)delegate
            {
                ReloadDeviceList();
            });
        }

        /// <summary>
        /// This method gets called when a ScripTouch device is removed from the system.
        /// </summary>
        /// <param name="path">Path of the removed device.</param>
        /// <param name="device">Device that was removed, may not be set.</param>
        public void ReceiveDeviceDetachedNotification(string path, Device device)
        {
            //Reload the device list on the main thread.
            this.Invoke((MethodInvoker)delegate
            {
                ReloadDeviceList();
            });
        }

        /// <summary>
        /// This method gets called any time there is a button event, there are four types
        /// of button events.
        ///    ButtonDown  - Pen is now against the glass on a button.
        ///    ButtonMove  - The pen has been down, and is moving on the surface of a button.
        ///    ButtonPress - The button has been pressed.
        ///    ButtonUp    - The button is no longer up, but has not been pressed.
        /// </summary>
        /// <param name="c">Button event</param>
        public void ReceiveButtonEvent(ButtonEvent c)
        {
            //Test to see if the button is pressed (it could also be a button down, button move or button up).
            if(c is ButtonPress) {
                if (c.GetRegion() == 3)
                {
                    //Ok Button, accept, but clear on next input.
                    clearOnNext = true;
                }
                else if (c.GetRegion() == 4)
                {
                    //Cancel Button, clear immediately.
                    Clear();
                }
            }
        }

        /// <summary>
        /// This method gets called when a coordinate is received from the device.
        /// </summary>
        /// <param name="c">Coordinate from the device.</param>
        public void ReceiveCoordinateEvent(Coordinate c)
        {
            //If the clear on next flag is set, clear the screen and the flag.
            if (clearOnNext)
            {
                Clear();
                clearOnNext = false;
            }

            //If we have a last coordinate draw a line from there to here.
            if (lastCoordinate != null)
            {
                graphics.DrawLine(pen, AdjustScale(lastCoordinate), AdjustScale(c));
                this.Invoke((MethodInvoker)delegate
                {
                    signature.Image = bitmap;
                });
            }

            //Test to see if the pen is down.
            if (c.IsPenDown())
            {
                //If so, save the last coordinate.
                lastCoordinate = c;
            }
            else
            {
                //If not, throw away the last coordinate to start a new stroke.
                lastCoordinate = null;
            }
        }

        /// <summary>
        /// This method uses the scaling information gathered from the device when
        /// we open it to scale the coordinate to our on-screen display.
        /// </summary>
        /// <param name="c">Coordinate from the device.</param>
        /// <returns>Point scaled to the on-screen display.</returns>
        private PointF AdjustScale(Coordinate c)
        {
            //Take both the X and Y coordinates and subtract the minimum range and divide by
            //the maximum range. This gives a number between 0 and 1 representing the percentage
            //of the width/height of the screen the pen is currently located in.
            double x = (c.GetX() - range.GetXMin()) / (double)range.GetXMax();
            double y = (c.GetY() - range.GetYMin()) / (double)range.GetYMax();

            //This will adjust the X and Y coordinates to fit in the designated box.
            x *= signature.Width;
            y *= signature.Height;

            //Create a new point.
            return new PointF((float)x, (float)y);
        }

        /// <summary>
        /// This method clears the signature display.
        /// </summary>
        public void Clear()
        {
            Brush brush = new SolidBrush(signature.BackColor);
            graphics.FillRectangle(brush, 0, 0, signature.Width, signature.Height);
            signature.Image = bitmap;

            if (connected && currentDevice != null)
            {
                currentDevice.ClearScreen();
            }
        }

        /// <summary>
        /// This method opens the device if it's not already open.
        /// </summary>
        private void OpenDevice()
        {
            if (!connected && currentDevice != null)
            {
                currentDevice.RegisterInputListener(this);
                currentDevice.Open();

                range = currentDevice.GetCoordinateRange();

                readThread = new Thread(ReadThread);
                readThread.Start();

                status.Text = "Connected";
                status.ForeColor = Color.Green;
                connectButton.Text = "Disconnect";
                connected = true;
            }
        }

        /// <summary>
        /// This method closes the device.
        /// </summary>
        private void CloseDevice()
        {
            if (currentDevice != null && currentDevice.IsOpen())
            {
                //Close the device.
                try
                {
                    currentDevice.Close();
                }
                catch (STException)
                {
                    //This can occur if the device has been removed from the system.
                }
            }
            status.Text = "Disconnected";
            currentDevice = null;
            status.ForeColor = Color.Red;
            connectButton.Text = "Connect";
            connected = false;
        }

        /// <summary>
        /// This method is the body for the reading thread. This method will
        /// call Read() on the open device until the device is no longer open
        /// or something happens to the device.
        /// </summary>
        private void ReadThread()
        {
            while (currentDevice != null && currentDevice.IsOpen())
            {
                try
                {
                    currentDevice.Read();
                }
                catch (STException)
                {
                    //This will get thrown in the device suddenly closes.
                    this.Invoke((MethodInvoker)delegate
                    {
                        CloseDevice();
                    });
                }
            }
        }

        /// <summary>
        /// Recieves ADC Channel Value scans, this is for diagnostics, we don't need this.
        /// </summary>
        /// <param name="c">Channel scan values.</param>
        public void ReceiveADCChannelValues(ADCChannelValues c)
        {
            //We aren't interested in this type of event.
        }

        /// <summary>
        /// Recieves ADC Scan Values, this is for diagnostics, we don't need this.
        /// </summary>
        /// <param name="c"></param>
        public void ReceiveADCScanValues(ADCScanValues c)
        {
            //We aren't interested in this type of event.
        }

        /// <summary>
        /// This gets called when a debug coordinate is detected from the device.
        /// </summary>
        /// <param name="c">Debug coordinate.</param>
        public void ReceiveDebugCoordinateEvent(DebugCoordinate c)
        {
            //We aren't interested in this type of event.   
        }

        /// <summary>
        /// This gets called when a magnetic card swipe is detected.
        /// </summary>
        /// <param name="c">Magnetic card swipe.</param>
        public void ReceiveMagneticStripEvent(MagneticCardSwipe c)
        {
            //We aren't interested in this type of event.
        }

        private void saveButton_Click(object sender, EventArgs e)
        {
            SaveFileDialog dialog = new SaveFileDialog();
            dialog.Filter = "PNG Files|*.png";
            if(dialog.ShowDialog() == DialogResult.OK) {
                signature.Image.Save(dialog.FileName, ImageFormat.Png);
            }
            
        }

        private void clearButton_Click(object sender, EventArgs e)
        {
            Clear();
        }
    }
}
