﻿// <copyright file="ConsoleExample.cs" company="Scriptel Corporation">
//      Copyright 2015 - Scriptel Corporation
// </copyright>
namespace ConsoleExample
{
    using System;
    using System.Collections.Generic;
    using ProScript;

    /// <summary>
    /// This class shows a very simple example of how to open and read from a
    /// ScripTouch device.
    /// </summary>
    class ConsoleExample : IDeviceNotificationListener, IInputListener
    {
        //Reference to the open device.
        private Device device;

        static void Main(string[] args)
        {
            //Create a new instance of this class so we can pass the references
            //to the ProScript library.
            ConsoleExample example = new ConsoleExample();

            //Register ourselves for hotplug notifications.
            DeviceManager.RegisterDeviceListener(example);

            //List the devices currently attached to the system.
            List<Device> devices = DeviceManager.GetAttachedDevices();

            //Print out the devices we found.
            Console.WriteLine("Found " + devices.Count + " device(s): ");
            foreach(Device d in devices) {
                Console.WriteLine("   " + d.GetModel());
            }
            Console.WriteLine();

            //Test to make sure we have at least one device.
            if (devices.Count > 0)
            {
                //Open the first device.
                Device device = devices[0];
                //Give a reference to our example instance.
                example.device = device;
                //Write out our intention to open the device.
                Console.WriteLine("Opening " + device.GetModel() + " (Sign / Swipe now)");
                //Register for input events.
                device.RegisterInputListener(example);
                //Open the device.
                device.Open();
                //While the device is open
                while (device.IsOpen())
                {
                    //Read from the device.
                    //This can be done from a separate thread if you don't
                    //want to block the main thread. See the FormExample for an
                    //example of this.
                    try
                    {
                        device.Read();
                    }
                    catch (STException e)
                    {
                        Console.WriteLine(e.Message);
                    }
                }
            }
            else
            {
                //If not, we'll just exit.
                Console.WriteLine("No devices found, exiting.");
            }
        }

        /// <summary>
        /// This method gets called when a new device is attached to the system.
        /// </summary>
        /// <param name="path">Path of the added device.</param>
        /// <param name="device">Reference to the added device.</param>
        public void ReceiveDeviceAttachedNotification(string path, Device device)
        {
            Console.WriteLine("Device added to the system: " + path);
        }

        /// <summary>
        /// This method gets called when a device gets removed from the system.
        /// </summary>
        /// <param name="path">Path of the removed device.</param>
        /// <param name="device">Reference to the removed device.</param>
        public void ReceiveDeviceDetachedNotification(string path, Device device)
        {
            Console.WriteLine("Device removed from the system: " + path);
        }

        /// <summary>
        /// This method gets called any time there is a button event, there are four types
        /// of button events.
        ///    ButtonDown  - Pen is now against the glass on a button.
        ///    ButtonMove  - The pen has been down, and is moving on the surface of a button.
        ///    ButtonPress - The button has been pressed.
        ///    ButtonUp    - The button is no longer up, but has not been pressed.
        /// </summary>
        /// <param name="c">Button event</param>
        public void ReceiveButtonEvent(ButtonEvent c)
        {
            Console.WriteLine("ButtonEvent: type=" + c.GetType() + ", x=" + c.GetX() + ", y=" + c.GetY() + ", region=" + c.GetRegion());

            if (c is ButtonPress && c.GetRegion() == 3)
            {
                //Ok was pressed, lets close.
                device.Close();
            }
        }

        /// <summary>
        /// This method gets called when a coordinate is received from the device.
        /// </summary>
        /// <param name="c">Coordinate from the device.</param>
        public void ReceiveCoordinateEvent(Coordinate c)
        {
            Console.WriteLine("Coordinate: x=" + c.GetX() + ", y=" + c.GetY() + ", penDown=" + c.IsPenDown());
        }

        /// <summary>
        /// This method gets called when a magnetic card swipe is detected from the device.
        /// </summary>
        /// <param name="c">Card swipe from the device.</param>
        public void ReceiveMagneticStripEvent(MagneticCardSwipe c)
        {
            //Print out the card swipe data.
            Console.WriteLine(c);
        }

        /// <summary>
        /// This is used for diagnostics, this isn't useful for most applications.
        /// </summary>
        /// <param name="c">Diagnostic value from the device.</param>
        public void ReceiveADCChannelValues(ADCChannelValues c)
        {
            //We don't care about this event.
        }

        /// <summary>
        /// This is used for diagnostics, this isn't useful for most applications.
        /// </summary>
        /// <param name="c">Diagnostic value from the device.</param>
        public void ReceiveADCScanValues(ADCScanValues c)
        {
            //We don't care about this event.
        }

        /// <summary>
        /// This is used for diagnostics, this isn't useful for most applications.
        /// </summary>
        /// <param name="c">Diagnostic value from the device.</param>
        public void ReceiveDebugCoordinateEvent(DebugCoordinate c)
        {
            //We don't care about this event.
        }
    }
}
